<?php

namespace App\Services\SalesOrder\Fulfillments;

use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentException;
use App\Integrations\Http\IntegrationTimeoutException;
use App\Managers\AddressManager;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Notifications\AutoFulfillmentFailedNotification;
use App\Repositories\IntegrationInstanceRepository;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Repositories\SettingRepository;
use Illuminate\Support\Facades\Notification;
use Throwable;

class AutomatedWarehouseFulfillment
{
    public function __construct(
        private readonly FulfillmentManager $manager,
        private readonly IntegrationInstanceRepository $instances,
        private readonly SalesOrderRepository $orders,
        private readonly SettingRepository $settings,
        private readonly AddressManager $addressManager,
    ) {
    }

    /**
     * @throws Throwable
     */
    public function fulfill(SalesOrder $order): void
    {
        if (!$order->shippingAddress || !$this->addressManager->isAddressValid($order->shippingAddress)) {
            customlog('SKU-6206', "$order->sales_order_number : AutomatedWarehouseFulfillment fails due to invalid shipping address", [], 7);

            return;
        }

        if ($order->salesChannel->integrationInstance->isLocal()) {
            customlog('SKU-6206', "$order->sales_order_number : AutomatedWarehouseFulfillment fails due to SKU.io manual order", [], 7);

            return;
        }

        // Entire order must be fulfillable in order to automate fulfillment
        if (! $order->is_fully_fulfillable) {
            customlog('SKU-6206', "$order->sales_order_number : AutomatedWarehouseFulfillment fails due to order not being fully fulfillable ", $order->salesOrderLines->pluck('product.sku')->toArray(), 7);

            return;
        }
        customlog('SKU-6206', "$order->sales_order_number : AutomatedWarehouseFulfillment proceeding (fully fulfillable) ", $order->salesOrderLines->pluck('product.sku')->toArray(), 7);


        if(!$order->shipping_method_id && $this->onlyFulfillWhenShippingMethodIsMapped()){
            customlog('SKU-6206', "$order->sales_order_number : AutomatedWarehouseFulfillment fails due to no shipping method specified and only fulfill when shipping method is mapped setting ", $order->salesOrderLines->pluck('product.sku')->toArray(), 7);
            return;
        }

        $automatedWarehouses = $this->instances->getAutoFulfillmentShippingProviders();

        if ($automatedWarehouses->isEmpty()) {
            customlog('SKU-6206', "$order->sales_order_number : AutomatedWarehouseFulfillment not proceeding due to no automated warehouses", [], 7);
            return;
        }
        if ($order->fulfillment_status == SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC) {
            customlog('SKU-6206', "$order->sales_order_number : AutomatedWarehouseFulfillment not proceeding due to order being out of sync", [], 7);
            return;
        }

        $applicableLines = $this->orders->getFulfillableLinesForOrder($order)->groupBy('warehouse_id');

        foreach ($applicableLines as $warehouseId => $lines) {
            $warehouse = $automatedWarehouses->firstWhere('warehouse_id', $warehouseId);
            $integration = $warehouse['integration'] ?? null;
            if ($integration == null) {
                continue;
            }

            $payload = [
                'warehouse_id' => $warehouseId,
                'fulfilled_at' => now(),
                'fulfillment_type' => mb_strtolower($integration['name']), // shipstation or starshipit
                'requested_shipping_method_id' => $order->shipping_method_id,
                'requested_shipping_method' => $order->requested_shipping_method,
            ];

            // add lines that not fulfilled yet
            $payload['fulfillment_lines'] = $lines->map(function (SalesOrderLine $salesOrderLine) {
                return [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => $salesOrderLine->unfulfilled_quantity,
                    'sales_channel_line_id' => $salesOrderLine->sales_channel_line_id,
                ];
            })->where('quantity', '>', 0)->values()->toArray();

            // all lines already fulfilled
            if (empty($payload['fulfillment_lines'])) {
                continue;
            }

            try {
                customlog('SKU-6206', "$order->sales_order_number : AutomatedWarehouseFulfillment fulfilling", [], 7);
                $this->manager->fulfill($order, $payload);
            }catch (SalesOrderFulfillmentDispatchException|IntegrationTimeoutException|SalesOrderFulfillmentException $e){
                customlog('auto_fulfillment_notification', "Failed fulfillment automated for {$order->sales_order_number}: {$e->getMessage()}", [
                    'order' => $order->sales_order_number,
                    'message' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                    'line' => $e->getLine(),
                    'file' => $e->getFile(),
                ], 7);
                if($this->notifiesAboutAutoFulfillmentErrors() && ($email = $this->getAutoFulfillmentEmail())){
                    // Send email to notify the SKU admin users about the failed auto fulfillment.
                    Notification::route('mail', $email)
                        ->notify(new AutoFulfillmentFailedNotification($order, $e->getMessage()));
                }
                customlog('auto_fulfillment_failure_notifications',
                    "Failed fulfillment automated for {$order->sales_order_number}: {$e->getMessage()}",
                    $e->getTrace(),
                    7
                );
            }
            catch (Throwable $e) {
                customlog('auto_fulfillment_failures',
                    "Failed fulfillment automated for {$order->sales_order_number}: {$e->getMessage()}"
                );
                if (config('app.env') != 'testing') {
                    throw $e;
                }
            }
        }
    }

    private function getAutoFulfillmentEmail(): ?string
    {
        return $this->settings->get(Setting::KEY_AUTO_FULFILLMENT_EMAIL);
    }

    private function onlyFulfillWhenShippingMethodIsMapped(): bool
    {
        return $this->settings->get(Setting::KEY_FULFILL_ON_MAPPED_SHIPPING_METHOD);
    }

    private function notifiesAboutAutoFulfillmentErrors(): bool
    {
        return $this->settings->get(Setting::KEY_AUTO_FULFILLMENT_FAILURE_NOTIFICATIONS);
    }
}
