<?php

namespace App\Services\SalesOrder\Fulfillments\Actions;

use App\Data\FulfillmentDataFromSalesOrdersResultData;
use App\Data\SalesOrderFulfillmentData;
use App\Data\SalesOrderFulfillmentLineData;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use Carbon\CarbonImmutable;
use Illuminate\Support\Collection;
use Illuminate\Support\Enumerable;

class BuildFulfillmentDataFromSalesOrders
{
    private array $errors = [];

    public function handle(Collection $salesOrders): FulfillmentDataFromSalesOrdersResultData
    {
        $salesOrderFulfillmentCollection = collect();

        $salesOrders = $this->sanitizeSalesOrdersForMissingLineReservations($salesOrders);

        $salesOrders->each(function (SalesOrder $salesOrder) use ($salesOrderFulfillmentCollection) {
            $salesOrderFulfillmentCollection->push($this->getFulfillmentDataCollectionFromSalesOrder($salesOrder));
        });

        return FulfillmentDataFromSalesOrdersResultData::from([
            'salesOrderFulfillments' => $salesOrderFulfillmentCollection->flatten(),
            'errors' => $this->errors,
        ]);
    }

    private function getFulfillmentDataCollectionFromSalesOrder(SalesOrder $salesOrder): Collection|Enumerable
    {
        $nextSequence = null;
        return SalesOrderFulfillmentData::collection($salesOrder
            ->salesOrderLines
            ->whereNotNull('warehouse_id')
            ->where('is_product', true)
            // Filter out sales order lines that are not fulfillable
            ->reject(fn(SalesOrderLine $salesOrderLine) => $salesOrderLine->fulfillable_quantity == 0)
            ->groupBy('warehouse_id')->map(function (Collection $lines, $warehouse_id) use ($salesOrder, &$nextSequence)
        {
            if (!$nextSequence)
            {
                $lastSequence = $salesOrder->salesOrderFulfillments->max(fn(SalesOrderFulfillment $fulfillment) => $fulfillment->fulfillment_sequence) ?? 0;
                $nextSequence = $lastSequence + 1;
            } else {
                $nextSequence++;
            }

            $fulfillmentType = $lines->first()->warehouse->order_fulfillment ?? SalesOrderFulfillment::TYPE_MANUAL;

            $linesCollection = SalesOrderFulfillmentLineData::collection($lines->map(function (SalesOrderLine $salesOrderLine) {
                return SalesOrderFulfillmentLineData::from([
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => $salesOrderLine->quantity,
                    'fulfillable_quantity' => $salesOrderLine->fulfillable_quantity,
                    'is_dropship' => $salesOrderLine->is_dropship,
                ]);
            }))->toArray();

            $shippingMethodFieldPrefix = $fulfillmentType === SalesOrderFulfillment::TYPE_MANUAL ? 'fulfilled' : 'requested';

           return SalesOrderFulfillmentData::from([
                'sales_order_id' => $salesOrder->id,
                'fulfillment_sequence' => $nextSequence,
                'warehouse_id' => $warehouse_id,
                'fulfillment_type' => $fulfillmentType,
                'fulfilled_at' => CarbonImmutable::now(),
                'fulfillment_lines' => $linesCollection,
                'user_id' => auth()->id(),
                'status' => SalesOrderFulfillment::getDefaultStatusByFulfillmentType($fulfillmentType),
                $shippingMethodFieldPrefix . '_shipping_method_id' => $salesOrder->shipping_method_id,
                $shippingMethodFieldPrefix . '_shipping_method_name' => $salesOrder->shippingMethod?->name,
            ]);

        })
            ->reject(fn(SalesOrderFulfillmentData $fulfillmentData) => count($fulfillmentData->fulfillment_lines) == 0)
        )->toCollection();
    }

    private function sanitizeSalesOrdersForMissingLineReservations(Collection $salesOrders): Collection
    {
        return $salesOrders->reject(function (SalesOrder $salesOrder)
        {
            $salesOrderLines = $salesOrder->salesOrderLines;
            foreach ($salesOrderLines as $salesOrderLine)
            {
                if ($salesOrderLine->is_dropship) {
                    continue;
                }
                $reservedQuantity = $salesOrderLine
                    ->inventoryMovements
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                    ->where('quantity', '>', 0)
                    ->where('layer_type', FifoLayer::class)
                    ->sum('quantity');

                $reservedQuantityForUnfulfilledLines = $reservedQuantity - $salesOrderLine->fulfilled_quantity;

                if ($salesOrderLine->fulfillable_quantity < $reservedQuantityForUnfulfilledLines) {
                    $this->errors[] = "Sales order {$salesOrderLine->salesOrder->sales_order_number} has fulfillable lines without reservations (data integrity issue)";
                    return true;
                }
            }
            return false;
        });
    }
}