<?php

namespace App\Services\SalesOrder;

use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Http\Requests\FulfillFBASalesOrderRequest;
use App\Http\Requests\FulfillSalesOrderRequest;
use App\Integrations\ShipStation;
use App\Jobs\Amazon\FBAOutboundCreateFulfillmentOrder;
use App\Managers\ProductInventoryManager;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\ShipStation\ShipstationOrder;
use App\Models\Warehouse;
use App\Notifications\MonitoringMessage;
use App\Response;
use App\SDKs\ShipStation\ShipStationException;
use App\Services\PurchaseOrder\ShipmentManager;
use App\Services\SalesOrderFulfillment\SubmitTrackingInfo;
use Carbon\Carbon;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Sonnenglas\AmazonMws\AmazonResponse;
use Throwable;

/**
 * @deprecated Use FulfillmentManager instead
 */
class FulfillSalesOrderService extends SalesOrderService
{
    /**
     * @var ShipmentManager
     */
    protected $shipmentManager;

    /**
     * FulfillSalesOrderService constructor.
     *
     *
     * @throws BindingResolutionException
     */
    public function __construct(SalesOrder $salesOrder)
    {
        $this->shipmentManager = app()->make(ShipmentManager::class);
        parent::__construct($salesOrder);
    }

    /**
     * Fulfill Sales Order.
     *
     *
     * @return SalesOrderFulfillment|array Array: error
     *
     * @throws Throwable
     */
    public function fulfill(
        FulfillSalesOrderRequest $request,
        $submitTrackingToSalesChannel = false,
        $submitToSalesProvider = true
    ) {
        return $this->fulfillWithInputs(
            $request->validated(),
            $submitTrackingToSalesChannel,
            $submitToSalesProvider
        );
    }

    /**
     * @return SalesOrderFulfillment|array Array: error
     *
     * @throws Throwable
     */
    public function fulfillWithInputs(
        array $inputs,
        bool $submitTrackingToSalesChannel = false,
        bool $submitToSalesProvider = true
    ) {
        // Only allow fulfillment of open orders
        if (! $this->salesOrder->isOpen()) {
            //throw new UnableToFulfillUnopenedOrderException('Only open sales orders can be fulfilled.');
        }

        $this->salesOrder->load(['salesOrderLines', 'salesOrderLines.product', 'salesOrderLines.inventoryMovements']);

        return DB::transaction(function () use ($submitTrackingToSalesChannel, $submitToSalesProvider, $inputs) {
            $fulfillment = $this->createFulfillment($inputs);

            /*
             * If the Fulfillment response is an array and not instanceof SalesOrderFulfillment
             * then we must rollback and return the array to the invoker.
             */
            if (is_array($fulfillment)) {
                DB::rollBack();
                customlog('starshipit', 'FulfillSalesOrderService::fulfillWithInputs createFulfillment', [
                    'fulfillment' => $fulfillment,
                ]);

                return $fulfillment; // error
            }

            /** @var Warehouse $warehouse */
            $warehouse = Warehouse::with([])->findOrFail($inputs['warehouse_id']);

            // if the warehouse is enable dropship, create purchase order shipment for lines
            if ($warehouse->is_dropship) {
                if (is_array($shipment = $this->createPurchaseOrderShipment($inputs, $warehouse, $fulfillment))) {
                    DB::rollBack();

                    return $shipment; // error
                }
            }

            // Submit order to ShipStation
            if ($fulfillment->fulfillment_type == SalesOrderFulfillment::TYPE_SHIPSTATION && $submitToSalesProvider) {
                $validationErrors = $fulfillment->toShipStationOrder()->validate();
                if (! empty($validationErrors)) {
                    DB::rollBack();
                    customlog('starshipit', 'FulfillSalesOrderService::fulfillWithInputs', [
                        'validationErrors' => $validationErrors,
                    ]);
                    return [__('messages.integration_instance.can_not_submit_order', ['resource' => 'ShipStation']), Response::CODE_INTERNAL_VALIDATION, '', $validationErrors];
                }

                try {
                    $this->submitToShipstation($fulfillment);
                } catch (ShipStationException $shipStationException) {
                    DB::rollBack();
                    // 401 Unauthorized
                    if ($shipStationException->getCode() == 401) {
                        IntegrationInstance::shipstation()->first()->unauthorizedConnection();
                    }

                    return [__('messages.integration_instance.can_not_submit_order', ['resource' => 'ShipStation']), Response::CODE_INTEGRATION_API_ERROR, 'id', (array) $shipStationException->getMessage()];
                }
            }

            // Submit order to ShipMyOrders

            // Submit order to Starshipit
            if ($fulfillment->fulfillment_type == SalesOrderFulfillment::TYPE_STARSHIPIT && $submitToSalesProvider) {
                $validationErrors = $fulfillment->toStarshipitOrder()->validate();
                if (! empty($validationErrors)) {
                    DB::rollBack();

                    return [__('messages.integration_instance.can_not_submit_order', ['resource' => 'Starshipit']), Response::CODE_INTERNAL_VALIDATION, '', $validationErrors];
                }

                $submitResponse = \App\Jobs\Starshipit\SubmitOrders::submitOrderFulfillmentToStarshipit($fulfillment);
                if (! $submitResponse['success']) {
                    DB::rollBack();
                    if ($submitResponse['error'] === 'unfulfillable') {
                        return [__('messages.integration_instance.unfulfillable_order', ['resource' => 'Starshipit']), Response::CODE_UNFULFILLABLE_TO_SHIPPING_PROVIDER, $submitResponse['error'], ['starshipit_order' => $submitResponse['order'], 'fulfillment_sequence' => $fulfillment->fulfillment_sequence]];
                    }
                    if ($submitResponse['error'] === 'archived') {
                        // recursive the function again with the next fulfillment sequence
                        $inputs['fulfillment_sequence'] = ++$fulfillment->fulfillment_sequence;

                        return $this->fulfillWithInputs($inputs, $submitTrackingToSalesChannel, $submitToSalesProvider);
                    }

                    return [__('messages.integration_instance.can_not_submit_order', ['resource' => 'Starshipit']), Response::CODE_INTEGRATION_API_ERROR, 'id', (array) $submitResponse['error']];
                }

                if (isset($submitResponse['linked']) && $submitResponse['linked']) {
                    $fulfillment->ssi_imported = true;
                }
            }

            // Submit tracking info to the sales channel
            if ($submitTrackingToSalesChannel && $fulfillment->status == SalesOrderFulfillment::STATUS_FULFILLED && ! $this->salesOrder->salesChannel->integrationInstance->isLocal()) {
                $response = SubmitTrackingInfo::factory($fulfillment)?->submit();
                if (! $response['success']) {
                    DB::rollBack();
                    $errorMessage = "Can't submit tracking to the sales channel: ".$this->salesOrder->sales_order_number.', response: '.json_encode($response);
                    Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($errorMessage));
                    Log::debug($errorMessage);

                    return [__('messages.integration_instance.can_not_submit_tracking', ['resource' => $this->salesOrder->salesChannel->integration->name]), Response::CODE_INTEGRATION_API_ERROR, 'id', (array) $response['message']];
                }
            }

            // Perform post fulfillment tasks
            $this->afterFulfillment($fulfillment);

            return $fulfillment;
        });
    }

    private function afterFulfillment(SalesOrderFulfillment $salesOrderFulfillment)
    {
        $this->updateFulfillmentStatus($salesOrderFulfillment->fulfilled_at ?? $this->salesOrder->fulfilled_at ?: null);
        // After the sales order fulfillment,
        // we send notification to the customer.
        //    if (  $this->salesOrder->customer->canNotifyAboutOrder() )
        //    {
        //      $this->salesOrder->customer->notify( new SalesOrderFulfilledNotification( $this->salesOrder ) );
        //    }
    }

    /**
     * Fulfill Sales Order.
     *
     *
     * @throws Throwable
     */
    public function fulfillFBA(FulfillFBASalesOrderRequest $request)
    {
        $this->salesOrder->load(['salesOrderLines', 'salesOrderLines.product', 'salesOrderLines.inventoryMovements']);

        DB::transaction(function () use ($request) {
            if (! $fulfillment = $this->createFBAFulfillment($request)) {
                DB::rollBack();
                Response::instance()->setMessage(__('messages.failed.restore'));

                return;
            }
            $integration = Warehouse::with([])->findOrFail($request->warehouse_id)->integrationInstance;

            /** @var AmazonResponse $createOrder */
            $createOrder = dispatch_sync(new FBAOutboundCreateFulfillmentOrder($integration, $fulfillment));
            if (! $createOrder->isSuccess()) {
                DB::rollBack();
                Response::instance()->error()->setMessage($createOrder->getError());

                return;
            }
            Response::instance()->addData([
                'resource' => 'sales order fulfillment',
                'id' => $fulfillment->id,
                'request_id' => $createOrder->getResult(),
            ]);

            // Perform post fulfillment tasks
            $this->afterFulfillment($fulfillment);

            Response::instance()->setMessage(__('messages.success.create', ['resource' => 'sales order fulfillment']));
        });
    }

    /**
     * @throws \App\SDKs\ShipStation\ShipStationException
     */
    public function submitToShipstation(SalesOrderFulfillment $fulfillment)
    {
        $integrationInstance = IntegrationInstance::with([])->whereHas('integration', function ($query) {
            $query->where('name', Integration::NAME_SHIPSTATION);
        })->first();

        $shipStation = app()->makeWith(ShipStation::class, ['integrationInstance' => $integrationInstance]);

        $response = $shipStation->submitOrder($fulfillment->toShipStationOrder());

        $result = $response->getResultAttribute();
        customlog('starshipit', 'FulfillSalesOrderService::submitToShipstation', [
            'response' => $response->toArray(),
        ]);

        $fulfillment->salesOrder()->update(['fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING]);
        ShipstationOrder::query()->where('orderId', $result->orderId)->firstOr(function () use ($result, $fulfillment) {
            ShipstationOrder::query()->create([
                'sku_fulfillment_id' => $fulfillment->id,
                'json_data' => $result->toArray(),
            ]);
        });
    }

    /**
     * Mark Sales Order as Fulfilled.
     */
    public function updateFulfillmentStatus(?Carbon $fulfilledDate = null)
    {
        $this->salesOrder->load([
            'salesOrderFulfillments',
            'salesOrderLines',
            'salesOrderLines.salesOrderFulfillmentLines',
        ]);

        // We want to update the status if a fulfillment is deleted so commenting out the below
        //        //Don't do anything for closed orders
        //        if ($this->salesOrder->order_status === SalesOrder::STATUS_CLOSED) {
        //            return;
        //        }

        if ($this->salesOrder->salesChannel->integrationInstance->isAmazonInstance() && $this->salesOrder->salesChannelOrder->FulfillmentChannel == 'AFN') {
            $this->salesOrder->order_status = SalesOrder::STATUS_CLOSED;
            $this->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_FULFILLED;
            $this->salesOrder->save();
            return;
        }

        if ($this->salesOrder->is_fully_fulfilled) {
            $this->salesOrder->close();
            $this->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_FULFILLED;
            $this->salesOrder->fulfilled_at = $fulfilledDate ?: now();
        } elseif ($this->salesOrder->is_over_fulfilled) {
            $this->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_OVER_FULFILLED;
            $this->salesOrder->close();
        } elseif ($this->salesOrder->is_awaiting_tracking) {
            $this->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING;
            $this->salesOrder->order_status = SalesOrder::STATUS_OPEN;
        } elseif ($this->salesOrder->is_partially_fulfilled) {
            $this->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_PARTIALLY_FULFILLED;
            $this->salesOrder->order_status = $this->salesOrder->canceled_at ? SalesOrder::STATUS_CLOSED : SalesOrder::STATUS_OPEN;
        } else {
            $this->salesOrder->order_status = $this->salesOrder->canceled_at ? SalesOrder::STATUS_CLOSED : SalesOrder::STATUS_OPEN;
            $this->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_UNFULFILLED;
            $this->salesOrder->fulfilled_at = null;
        }

        $this->salesOrder->save();
    }

    /**
     * Fully fulfill closed sales order.
     *
     *
     * @throws Throwable
     */
    public function fullyFulfill(?string $trackingNumber = null, $submitTrackingToSalesChannel = false)
    {
        $this->salesOrder->load('salesOrderLines', 'salesOrderLines.warehouse');

        DB::transaction(function () use ($submitTrackingToSalesChannel, $trackingNumber) {
            // fulfill products only
            $fulfillableLines = $this->salesOrder->salesOrderLines()->where('is_product', true)->get();

            // sales order fulfillment for every warehouse
            foreach ($fulfillableLines->groupBy('warehouse_id') as $warehouseId => $lines) {
                if ($warehouseId) {
                    if (! ($fulfillment = $this->createFulfillmentFromOriginalLines($lines, $trackingNumber))) {
                        DB::rollBack();

                        return;
                    }

                    /** @var Warehouse $warehouse */
                    $warehouse = $lines->first()->warehouse;

                    // dropship
                    if ($warehouse->is_dropship) {
                        if (! $this->createPOShipmentFromOriginalLines($lines, $trackingNumber)) {
                            DB::rollBack();

                            return;
                        }

                        $this->createPurchaseInvoiceForDropship($lines);
                    }
                    // submit tracking info to the sales channel
                    if ($submitTrackingToSalesChannel && $fulfillment->wasRecentlyCreated && $fulfillment->status == SalesOrderFulfillment::STATUS_FULFILLED && ! $this->salesOrder->salesChannel->integrationInstance->isLocal()) {
                        $response = SubmitTrackingInfo::factory($fulfillment)?->submit();
                        if (! $response['success']) {
                            DB::rollBack();
                            $errorMessage = "Can't submit tracking to the sales channel: ".$this->salesOrder->sales_order_number.', response: '.json_encode($response);
                            Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage($errorMessage));
                            Log::debug($errorMessage);

                            Response::instance()->addError(__('messages.integration_instance.can_not_submit_tracking', ['resource' => $this->salesOrder->salesChannel->integration->name]), Response::CODE_INTEGRATION_API_ERROR, 'id', (array) $response['message']);
                        }
                    }

                    $this->afterFulfillment($fulfillment);
                }
            }
        });
    }

    /**
     * Create Sales Order Fulfillment.
     *
     *
     * @return SalesOrderFulfillment|array Array: error
     *
     * @throws \App\Exceptions\InsufficientStockException
     */
    private function createFulfillment(array $inputs)
    {
        // New Sales Order Fulfillment
        $salesOrderFulfillment = new SalesOrderFulfillment($inputs);

        $this->salesOrder->salesOrderFulfillments()->save($salesOrderFulfillment);

        $is_fulfilled_pre_audit_trail = false;

        /*
         * Fulfillment date represents when inventory movements are made.  Right now we make inventory movements when the
         * fulfilment record is actually created.  If we are importing a historical order, created_at would be defined here
         * as the date of the actual fulfillment (i.e. from sales channel).
         *
         * This process would be refactored after SKU-4065 where we will be removing the ability to create externally fulfilled
         * fulfillments upon order creation.  Instead, we would use externally fulfilled quantity.
         */
        $fulfillment_date = $salesOrderFulfillment->created_at ?? Carbon::now('utc');

        if (isset($this->salesOrder->salesChannel->integrationInstance->audit_trail_start_date)) {
            $integrationInstance = $this->salesOrder->salesChannel->integrationInstance;
            $is_fulfilled_pre_audit_trail = $salesOrderFulfillment->fulfilled_at->lt(Carbon::parse($integrationInstance->audit_trail_start_date->setTimezone(('UTC')))) ?? false;
        }

        foreach ($inputs['fulfillment_lines'] as $index => $fulfillmentLine) {
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = $this->salesOrder->salesOrderLines->firstWhere('id', $fulfillmentLine['sales_order_line_id']);

            // Remove lines with no audit trails
            if ($salesOrderLine->no_audit_trail && $is_fulfilled_pre_audit_trail) {
                continue;
            }

            // Remove lines with active backorder queues.
            $fulfillableQuantity = max(0, $salesOrderLine->unfulfilled_quantity - $salesOrderLine->active_backordered_quantity);

            if ($fulfillableQuantity == 0) {
                continue;
            }

            // New Sales Order Fulfillment Line.
            $salesOrderFulfillmentLine = new SalesOrderFulfillmentLine();
            $salesOrderFulfillmentLine->sales_order_line_id = $salesOrderLine->id;
            $salesOrderFulfillmentLine->quantity = min($fulfillmentLine['quantity'], $fulfillableQuantity);
            $salesOrderFulfillment->salesOrderFulfillmentLines()->save($salesOrderFulfillmentLine);
            $salesOrderFulfillment->setRelation('salesOrderFulfillment', $salesOrderFulfillment);

            $salesOrderFulfillmentLine->salesOrderLine->fulfilled_quantity += $fulfillmentLine['quantity'];

            $salesOrderFulfillmentLine->salesOrderLine->update();

            if ($salesOrderLine->product && ! $salesOrderLine->is_dropship) {
                $lineReversed = $salesOrderFulfillmentLine->negateReservationMovements();
                if (! $lineReversed) {
                    //                    dd($lineReversed, $salesOrderLine->toArray(), $salesOrderLine->product->sku);
                    return [__('messages.sales_order.no_reservation_movements'), 'SalesOrderLine'.Response::CODE_RESERVE_NOT_FOUND, "fulfillment_lines.{$index}.sales_order_line_id"];
                }
            }
        }

        if ($salesOrderFulfillment->salesOrderFulfillmentLines()->count() == 0) {
            /**
             * None of the sales order line was actually fulfillable, we
             * cancel the fulfillment.
             */
            customlog('SKU-6191', 'FulfillSalesOrderService::484 deleting '.$salesOrderFulfillment->fulfillment_number.' and ignoring shipping provider exceptions', [
                'debug' => debug_pretty_string(),
            ], 7);
            $salesOrderFulfillment->delete(true);

            return [
                __('messages.sales_order.is_backordered'),
                'SalesOrder'.Response::CODE_UNFULFILLABLE_TO_SHIPPING_PROVIDER,
                "Sales order: {$this->salesOrder->sales_order_number}.",
            ];
        }

        // mark sales order as fulfilled if fulfillment type is manual
        if ($salesOrderFulfillment->fulfillment_type == SalesOrderFulfillment::TYPE_MANUAL) {
            $this->updateFulfillmentStatus($salesOrderFulfillment->fulfilled_at);
        }

        // cache inventory
        $productIds = SalesOrderLine::query()->whereIn('id', collect($inputs['fulfillment_lines'])->pluck('sales_order_line_id'))->pluck('product_id')->toArray();
        (new ProductInventoryManager($productIds))->setUpdateAverageCost(false)->updateProductInventoryAndAvgCost();

        return $salesOrderFulfillment;
    }

    /**
     * Create Sales Order Fulfillment.
     *
     *
     * @return bool | SalesOrderFulfillment
     */
    private function createFBAFulfillment(FulfillFBASalesOrderRequest $request): bool
    {
        $inputs = $request->validated();

        // New Sales Order Fulfillment
        $salesOrderFulfillment = new SalesOrderFulfillment($inputs);
        $salesOrderFulfillment->fulfillment_type = SalesOrderFulfillment::TYPE_FBA;
        $salesOrderFulfillment->fulfilled_at = now();
        $salesOrderFulfillment->metadata = [
            'shipping_speed' => $request->input('shipping_speed'),
            'comments' => $request->input('comments'),
        ];
        $this->salesOrder->salesOrderFulfillments()->save($salesOrderFulfillment);

        foreach ($inputs['fulfillment_lines'] as $index => $fulfillmentLine) {
            $salesOrderLine = $this->salesOrder->salesOrderLines->firstWhere('id', $fulfillmentLine['sales_order_line_id']);

            // New Sales Order Fulfillment Line.
            $salesOrderFulfillmentLine = new SalesOrderFulfillmentLine();
            $salesOrderFulfillmentLine->sales_order_line_id = $salesOrderLine->id;
            $salesOrderFulfillmentLine->quantity = $fulfillmentLine['quantity'];
            $productListing = ProductListing::with([])->find($fulfillmentLine['product_listing_id']);
            $salesOrderFulfillmentLine->metadata = [
                'product_listing_id' => $productListing->id,
                'listing_sku' => $productListing->listing_sku,
            ];
            $salesOrderFulfillment->salesOrderFulfillmentLines()->save($salesOrderFulfillmentLine);
            $salesOrderFulfillment->setRelation('salesOrderFulfillment', $salesOrderFulfillment);
        }

        return $salesOrderFulfillment;
    }

    private function createFulfillmentFromOriginalLines(Collection $lines, ?string $trackingNumber = null)
    {
        /** @var Warehouse $warehouse */
        $warehouse = $lines->first()->warehouse;
        if ($lines->where('unfulfilled_quantity', '>', 0)->isEmpty()) {
            /*
             * TODO: I'm not sure if this if statement is the right solution.  I think the problem is with the previous
             *  line... it is passing even when there are no fulfillments
             */
            if ($this->salesOrder->salesOrderFulfillments->isNotEmpty()) {
                return $this->salesOrder->salesOrderFulfillments()->where('warehouse_id', $warehouse->id)->firstOrFail();
            }
        }

        // New Sales Order Fulfillment
        $salesOrderFulfillment = new SalesOrderFulfillment();
        $salesOrderFulfillment->fulfillment_type = SalesOrderFulfillment::TYPE_MANUAL;
        $salesOrderFulfillment->warehouse_id = $warehouse->id;
        $salesOrderFulfillment->tracking_number = $trackingNumber;
        $salesOrderFulfillment->requested_shipping_method_id = $this->salesOrder->shipping_method_id;
        $salesOrderFulfillment->requested_shipping_method = $this->salesOrder->requested_shipping_method;
        $salesOrderFulfillment->fulfilled_at = $this->salesOrder->fulfilled_at ?: now();
        $this->salesOrder->salesOrderFulfillments()->save($salesOrderFulfillment);

        // add lines
        foreach ($lines as $line) {
            $salesOrderFulfillmentLine = new SalesOrderFulfillmentLine();
            $salesOrderFulfillmentLine->sales_order_line_id = $line->id;
            $salesOrderFulfillmentLine->quantity = $line->unfulfilled_quantity;
            $salesOrderFulfillment->salesOrderFulfillmentLines()->save($salesOrderFulfillmentLine);
            $salesOrderFulfillmentLine->setRelation('salesOrderFulfillment', $salesOrderFulfillment);

            if ($line->product && ! $line->is_dropship) {
                $lineReversed = $salesOrderFulfillmentLine->negateReservationMovements();
                if (! $line->salesOrder->is_fba && ! $lineReversed) {
                    Response::instance()->addError(__('messages.sales_order.no_reservation_movements'), 'SalesOrderLine'.Response::CODE_RESERVE_NOT_FOUND, "fulfillment_lines.{$line->id}.sales_order_line_id");

                    return false;
                }
            }
        }

        return $salesOrderFulfillment;
    }

    /**
     * Create Purchase Order Shipment.
     *
     *
     * @return PurchaseOrderShipment|array Array: error
     *
     * @throws NotOpenPurchaseOrderException
     */
    public function createPurchaseOrderShipment(
        array $inputs,
        Warehouse $warehouse,
        ?SalesOrderFulfillment $fulfillment
    ) {
        // get purchase order by "supplier_warehouse_id"
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = $this->salesOrder->purchaseOrders()
            ->where('supplier_warehouse_id', $warehouse->id)
            ->with(['purchaseOrderLines'])
            ->first();
        if (! $purchaseOrder) {
            return [__('messages.sales_order.no_dropship_purchase_order'), 'SalesOrder'.Response::CODE_DROPSHIP_PO_NOT_FOUND, 'sales_order_id'];
        }
        // purchase order still not approved
        if ($purchaseOrder->isDraft()) {
            return [__('messages.sales_order.draft_dropship_purchase_order'), 'SalesOrder'.Response::CODE_DROPSHIP_PO_DRAFT, 'sales_order_id'];
        }

        // create Purchase Order Shipment
        /** @var PurchaseOrderShipment $purchaseOrderShipment */
        $inputs['sales_order_fulfillment_id'] = $fulfillment->id;
        $inputs['fulfilled_shipping_method_id'] = $fulfillment->fulfilled_shipping_method_id ?: $fulfillment->requested_shipping_method_id;
        if (empty($inputs['fulfilled_shipping_method_id'])) {
            $inputs['fulfilled_shipping_method'] = $fulfillment->fulfilled_shipping_method ?: $fulfillment->requested_shipping_method;
        }
        $purchaseOrderShipment = $purchaseOrder->purchaseOrderShipments()->create($inputs);
        $receiptLines = [];

        foreach ($inputs['fulfillment_lines'] as $index => $fulfillmentLine) {
            $salesOrderLine = $this->salesOrder->salesOrderLines->firstWhere('id', $fulfillmentLine['sales_order_line_id']);

            // dropship
            $purchaseOrderLine = $purchaseOrder->purchaseOrderLines->firstWhere('product_id', $salesOrderLine->product_id);
            // add Purchase Order Shipment lines
            $purchaseOrderShipment->purchaseOrderShipmentLines()->create([
                'purchase_order_line_id' => $purchaseOrderLine->id,
                'quantity' => $fulfillmentLine['quantity'],
            ]);

            // Add purchase order receipts (SKU-2466)
            $receiptLines[] = [
                'purchase_order_line_id' => $purchaseOrderLine->id,
                'quantity' => $fulfillmentLine['quantity'],
            ];
        }

        // Receive the purchase order lines
        $payload = [
            'purchase_order_shipment_id' => $purchaseOrderShipment->id,
            'received_at' => now(),
        ];
        $this->shipmentManager->receiveLines($receiptLines, $payload, false);

        // mark Purchase Order as shipped
        $purchaseOrder->approve($purchaseOrderShipment->shipment_date);
        $purchaseOrder->shipped($purchaseOrderShipment->shipment_date, true);

        return $purchaseOrderShipment;
    }

    /**
     * @param  SalesOrderLine[]  $lines
     */
    private function createPOShipmentFromOriginalLines(array $lines, ?string $trackingNumber = null): bool
    {
        /** @var Warehouse $warehouse */
        $warehouse = $lines->first()->warehouse;

        // get purchase order by "supplier_warehouse_id"
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = $this->salesOrder->purchaseOrders()
            ->where('supplier_warehouse_id', $warehouse->id)
            ->with(['purchaseOrderLines'])
            ->first();
        if (! $purchaseOrder) {
            Response::instance()->addError(__('messages.sales_order.no_dropship_purchase_order'), 'SalesOrder'.Response::CODE_DROPSHIP_PO_NOT_FOUND, 'sales_order_id');

            return false;
        }

        // create Purchase Order Shipment
        $purchaseOrderShipment = new PurchaseOrderShipment();
        $purchaseOrderShipment->shipment_date = $this->salesOrder->fulfilled_at;
        $purchaseOrderShipment->fulfilled_shipping_method_id = $this->salesOrder->shipping_method_id;
        $purchaseOrderShipment->tracking = $trackingNumber;
        $purchaseOrder->purchaseOrderShipments()->save($purchaseOrderShipment);

        foreach ($lines as $line) {
            $salesOrderLine = $this->salesOrder->salesOrderLines->firstWhere('id', $line->id);

            $purchaseOrderLine = $purchaseOrder->purchaseOrderLines->firstWhere('product_id', $salesOrderLine->product_id);
            // add Purchase Order Shipment lines
            $purchaseOrderShipment->purchaseOrderShipmentLines()->create([
                'purchase_order_line_id' => $purchaseOrderLine->id,
                'quantity' => $line->unfulfilled_quantity,
            ]);
        }

        // mark Purchase Order as shipped
        $purchaseOrder->approve($purchaseOrderShipment->shipment_date);
        $purchaseOrder->shipped($purchaseOrderShipment->shipment_date, true);

        return true;
    }

    private function createPurchaseInvoiceForDropship($lines)
    {
        /** @var Warehouse $warehouse */
        $warehouse = $lines->first()->warehouse;

        // get purchase order by "supplier_warehouse_id"
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = $this->salesOrder->purchaseOrders()
            ->where('supplier_warehouse_id', $warehouse->id)
            ->with(['purchaseOrderLines'])
            ->first();
        if (! $purchaseOrder) {
            Response::instance()->addError(__('messages.sales_order.no_dropship_purchase_order'), 'SalesOrder'.Response::CODE_DROPSHIP_PO_NOT_FOUND, 'sales_order_id');

            return false;
        }

        $purchaseInvoice = new PurchaseInvoice();
        $purchaseInvoice->purchase_invoice_date = $this->salesOrder->fulfilled_at;
        $purchaseInvoice->supplier_id = $warehouse->supplier_id;
        $purchaseOrder->purchaseInvoices()->save($purchaseInvoice);

        $purchaseInvoiceLines = $purchaseOrder->purchaseOrderLines->map(function (PurchaseOrderLine $purchaseOrderLine) {
            return [
                'purchase_order_line_id' => $purchaseOrderLine->id,
                'quantity_invoiced' => $purchaseOrderLine->quantity,
            ];
        });

        $purchaseInvoice->purchaseInvoiceLines()->createMany($purchaseInvoiceLines->toArray());

        $purchaseOrder->invoiced($purchaseInvoice->purchase_invoice_date);

        return true;
    }
}
