<?php

namespace App\Services\SalesOrder\Concerns;

use App\Exceptions\SalesOrder\SalesOrderFulfillmentException;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderShipment;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Services\PurchaseOrder\ShipmentManager;
use Exception;
use Illuminate\Support\Collection;
use Throwable;

trait ManagesDropshipOrders
{
    /**
     * @throws Throwable
     */
    private function createDropshipPurchaseOrdersFor(SalesOrder $order): SalesOrder
    {
        $order->dropshipLines->groupBy('warehouse_id')->each(/**
         * @throws Throwable
         */ function ($salesOrderLines, $warehouseId) use ($order) {
            $this->createDropshipPurchaseOrderAtWarehouse($warehouseId, $order, $salesOrderLines);
        });

        // We update the fulfillment status of the order
        // after the dropship purchase order is processed.
        $order->updateFulfillmentStatus();

        return $order;
    }

    /**
     * @throws Throwable
     */
    private function handleDropshipInOrderUpdate(SalesOrder $order): SalesOrder
    {
        if (! $this->orderHasDropshippableLines($order)) {
            // The order has no dropshippable lines.
            // If there are existing dropship POs attached to the order,
            // we remove those that aren't submitted. For submitted/finalized dropship
            // orders, it's the user's responsibility to decide on what to do.

            return $order->hasDropshipOrders() ?
                $order->removeUnsubmittedPurchaseOrders() :
                $order;
        }

        // At this point, the sales order has dropship lines.
        // In case the order doesn't already have dropship
        // orders, we create them.
        if (! $order->hasDropshipOrders()) {
            // We create the dropship PO
            return $this->createDropshipPurchaseOrdersFor($order);
        }

        // Next, the order already has dropship purchase orders.
        // We need to sync those dropship orders with the dropship
        // lines on the sales order for each warehouse.
        $dropshipOrders = [];
        $order->dropshipLines->groupBy('warehouse_id')->each(/**
         * @throws Throwable
         */ function ($salesOrderLines, $warehouseId) use ($order, &$dropshipOrders) {
            if (! is_null($dropshipOrder = $this->syncDropshipPurchaseOrderAtWarehouse($warehouseId, $order, $salesOrderLines))) {
                $dropshipOrders[] = $dropshipOrder->id;
            }
        });

        // We sync the dropship orders to only what is retained.
        $order->syncDropshipOrders($dropshipOrders);

        // We update the fulfillment status of the order
        // after the dropship purchase order is processed.
        $order->updateFulfillmentStatus();

        return $order;
    }

    /**
     * @throws Throwable
     */
    private function createDropshipPurchaseOrderAtWarehouse(
        int $warehouseId,
        SalesOrder $order,
        Collection $salesOrderLines,
    ): ?PurchaseOrder {
        if (! ($warehouse = $this->getAutoFulfillableDropshipWarehouseById($warehouseId, $order))) {
            // The supplier attached to the warehouse doesn't automatically fulfill dropship orders.
            return null;
        }

        $purchaseOrder = $this->purchaseOrders->createDropshipPurchaseOrder(
            salesOrder: $order,
            warehouse: $warehouse,
            salesOrderLines: $salesOrderLines
        );

        // We approve the dropship purchase order.
        $purchaseOrder->approve();

        if ($warehouse->supplier->auto_submit_dropship_po) {
            $purchaseOrder->submit(
                approvingDropship: true,
            );
        }

        return $purchaseOrder;
    }

    /**
     * @throws Throwable
     */
    private function syncDropshipPurchaseOrderAtWarehouse(
        int $warehouseId,
        SalesOrder $order,
        Collection $salesOrderLines
    ): ?PurchaseOrder {
        if (! ($warehouse = $this->getAutoFulfillableDropshipWarehouseById($warehouseId, $order))) {
            // The supplier attached to the warehouse doesn't automatically fulfill dropship orders.
            return null;
        }

        $existingDropshipOrder = $order->getDropshipOrderAtWarehouse($warehouse);
        if (! $existingDropshipOrder) {
            // We create the dropship order.
            return $this->createDropshipPurchaseOrderAtWarehouse($warehouseId, $order, $salesOrderLines);
        }

        if (! $existingDropshipOrder->unsubmitted) {
            // The dropship purchase order is already submitted,
            // we leave it to the user to handle the updates.
            return null;
        }

        // We update the with the lines in the sales order.
        $existingDropshipOrder->setPurchaseOrderLines(
            purchaseOrderLines: $salesOrderLines->map(function (SalesOrderLine $line) use ($existingDropshipOrder) {
                $purchaseOrderLine = $existingDropshipOrder->purchaseOrderLines->firstWhere('product_id', $line->product_id);

                return $line->only(['description', 'product_id', 'amount', 'quantity']) +
                    ['id' => $purchaseOrderLine?->id ?? null];
            }),
            sync: false
        );

        // We re-approve the purchase order
        $existingDropshipOrder->approve();

        // Finally, we submit the dropship order to the supplier if
        // we can auto-submit to the supplier.
        if ($warehouse->supplier->auto_submit_dropship_po) {
            $existingDropshipOrder->submit(
                notifySupplier: true,
                approvingDropship: true,
            );
        }

        return $existingDropshipOrder;
    }

    /**
     * @throws Exception|Throwable
     */
    protected function shipDropshipPurchaseOrders(
        SalesOrderFulfillment $fulfillment,
        Warehouse $warehouse,
        array $payload = []
    ): void {
        $purchaseOrder = $this->validateDropshipPurchaseOrder(
            $this->purchaseOrders->findDropshipOrder($fulfillment->salesOrder, $warehouse),
            $fulfillment
        );

        $payload['sales_order_fulfillment_id'] = $fulfillment->id;

        // We create the shipment and receive it
        $shipment = $this->purchaseOrders->createShipment(
            purchaseOrder: $purchaseOrder,
            payload: $this->makeShipmentPayload($purchaseOrder, $fulfillment, $payload)
        );

        // Receive the shipment.
        /** @var ShipmentManager $manager */
        $manager = app(ShipmentManager::class);
        $manager->receiveShipment(
            payload: $this->makeReceiptPayload($shipment, $warehouse),
            addToInventory: false
        );

        // Update the shipment status on the PO
        $purchaseOrder->shipped(shippedDate: $shipment->shipment_date, dropship: true);

        $fulfillment->salesOrderFulfillmentLines->each(function (SalesOrderFulfillmentLine $fulfillmentLine) {
            $fulfillmentLine->salesOrderLine->incrementFulfilledQuantity($fulfillmentLine->quantity);
        });
        $fulfillment->salesOrder->updateFulfillmentStatus();
    }

    private function makeReceiptPayload(PurchaseOrderShipment $shipment, Warehouse $warehouse): array
    {
        return [
            'purchase_order_shipment_id' => $shipment->id,
            'received_at' => $shipment->shipment_date,
            'warehouse_id' => $warehouse->id,
        ];
    }

    private function makeShipmentPayload(
        PurchaseOrder $purchaseOrder,
        SalesOrderFulfillment $fulfillment,
        array $payload
    ): array {
        $payload['sales_order_fulfillment_id'] = $fulfillment->id;
        $payload['fulfilled_shipping_method_id'] = $fulfillment->fulfilled_shipping_method_id ?:
            $fulfillment->requested_shipping_method_id;
        if (empty($payload['fulfilled_shipping_method_id'])) {
            $payload['fulfilled_shipping_method'] = $fulfillment->fulfilled_shipping_method ?:
                $fulfillment->requested_shipping_method;
        }

        $payload['purchase_order_shipment_lines'] = $fulfillment->salesOrderFulfillmentLines->map(
            function (SalesOrderFulfillmentLine $fulfillmentLine) use ($purchaseOrder) {
                return [
                    'purchase_order_line_id' => $purchaseOrder->purchaseOrderLines->firstWhere('product_id',
                        $fulfillmentLine->salesOrderLine->product_id)?->id,
                    'quantity' => $fulfillmentLine->quantity,
                ];
            })
            ->filter(fn ($shipmentLine) => ! is_null($shipmentLine['purchase_order_line_id']))
            ->toArray();

        if(empty($payload['shipment_date'])){
            $payload['shipment_date'] = $fulfillment->fulfilled_at;
        }

        return $payload;
    }

    /**
     * @throws SalesOrderFulfillmentException
     */
    private function validateDropshipPurchaseOrder(
        ?PurchaseOrder $purchaseOrder,
        SalesOrderFulfillment $fulfillment
    ): PurchaseOrder {
        // Dropship order must exist.
        if (! $purchaseOrder) {
            throw new SalesOrderFulfillmentException(
                salesOrder: $fulfillment->salesOrder,
                fulfillment: $fulfillment,
                message: __('messages.sales_order.no_dropship_purchase_order'),
            );
        }

        // Dropship order must be approved.
        if ($purchaseOrder->isDraft()) {
            throw new SalesOrderFulfillmentException(
                salesOrder: $fulfillment->salesOrder,
                fulfillment: $fulfillment,
                message: __('messages.sales_order.draft_dropship_purchase_order')
            );
        }

        return $purchaseOrder;
    }

    /**
     * @throws Throwable
     */
    private function getAutoFulfillableDropshipWarehouseById(
        int $warehouseId,
        SalesOrder $order
    ): ?Warehouse {
        $warehouse = $this->warehouses->findById($warehouseId);
        throw_if(! $warehouse->is_dropship || ! $warehouse->isSupplierWarehouse(),
            'Dropship Purchase order for '.$order->sales_order_number.' must be to a dropship enabled supplier warehouse.');

        return $warehouse->supplier->auto_fulfill_dropship ? $warehouse : null;
    }

    public function orderHasDropshippableLines(SalesOrder $order): bool
    {
        return $order->dropshipLines->isNotEmpty();
    }
}
