<?php

namespace App\Services\SalesOrder;

use App\Exceptions\ActionUnavailableTemporaryException;
use App\Http\Requests\DropshipSalesOrderRequest;
use App\Http\Requests\FulfillSalesOrderRequest;
use App\Integrations\Starshipit;
use App\Jobs\Starshipit\GetTrackingJob;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Starshipit\StarshipitOrder;
use App\Models\Warehouse;
use App\Response;
use App\SDKs\Starshipit\StarshipitResponse;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Laravel\Telescope\Telescope;

class BulkFulfillSalesOrdersService
{
    const MAX_SYNC_COUNT = 1000;

    const MAX_SHIPSTATION_ORDERS_COUNT = 100; // 100 orders per request

    const MAX_STARSHIPIT_ORDERS_COUNT = 50; // 50 orders per request

    protected $request;

    private $count;

    private $fulfillments = [SalesOrderFulfillment::TYPE_SHIPSTATION => [], SalesOrderFulfillment::TYPE_STARSHIPIT => []];

    private $response = [];

    private $errorsCount = 0;

    private $attempts = 0;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    /**
     * @return array array of errors
     *
     * @throws \Throwable
     */
    public function fulfill(): array
    {
        set_time_limit(0);

        try {
            $this->getBuilder()->each(function (SalesOrder $salesOrder) {
                if (! $salesOrder->isOpen()) {
                    $this->response[$salesOrder->id][] = Response::getError(__('messages.sales_order.is_not_open'), 'SalesOrder'.Response::CODE_IS_NOT_OPEN, 'id');
                    $this->errorsCount++;

                    return; // continue
                }
                // grouped lines by warehouse
                $groupedLines = $salesOrder->salesOrderLines->where('is_product', true)->where('no_audit_trail', 0)->where('quantity', '>', 0)->groupBy('warehouse_id');
                // each group in a sales order fulfillment
                $groupedLines->each(function (Collection $salesOrderLines, $warehouseId) use ($salesOrder) {
                    // the sales order has unmapped products
                    if (empty($warehouseId)) {
                        $this->response[$salesOrder->id][] = Response::getError(__('messages.sales_order.has_unmapped_lines'), 'SalesOrder'.Response::CODE_HAS_UNMAPPED_PRODUCTS, 'id');
                        $this->errorsCount++;

                        return; // continue
                    }

                    /* @var Warehouse $warehouse */
                    $warehouse = $salesOrderLines->first()->warehouse;
                    try {
                        if ($warehouse->is_dropship) {
                            $fulfillRequest = $this->getDropshipSalesOrderRequest($salesOrder, $salesOrderLines);
                        } elseif ($warehouse->isAmazonFBA()) {
                            // TODO: implement FulfillFBASalesOrderRequest
                            return; // continue
                        } else {
                            $fulfillRequest = $this->getFulfillmentSalesOrderRequest($salesOrder, $salesOrderLines, $this->request->input('fulfilled_at', now()));
                        }
                    } catch (ValidationException $validationException) {
                        $this->response[$salesOrder->id][$warehouseId] = Response::getError('', Response::CODE_INTERNAL_VALIDATION, '', $validationException->errors());
                    } catch (ActionUnavailableTemporaryException $unavailableTemporaryException) {
                        $this->response[$salesOrder->id][$warehouseId] = $unavailableTemporaryException->getResponseError('id');
                    }

                    // fulfilled, unfulfillable or dropship requested
                    if (! isset($fulfillRequest) || $fulfillRequest === true) {
                        return; // continue
                    }

                    if ($warehouse->is_dropship) {
                        // dropship request(create a purchase order)
                        ApproveSalesOrderService::make($salesOrder)->approveDropshipRequest($fulfillRequest);
                        $this->response[$salesOrder->id][$warehouseId] = __('messages.success.create', ['resource' => 'dropship request']);
                    } else {
                        // fulfill error
                        // TODO: after implementing shipstation
                        if (is_array($fulfill = FulfillSalesOrderService::make($salesOrder)->fulfill($fulfillRequest, true, $fulfillRequest->input('fulfillment_type') !== SalesOrderFulfillment::TYPE_STARSHIPIT))) {
                            $this->response[$salesOrder->id][$warehouseId] = Response::getError(...$fulfill);
                            $this->errorsCount++;
                        } else {
                            $this->fulfillments[$fulfill->fulfillment_type][] = $fulfill;
                            $this->response[$salesOrder->id][$warehouseId] = __('messages.success.create', ['resource' => "sales order fulfillment '{$fulfill->fulfillment_number}'"]);
                        }
                    }
                    // submit fulfillments if their count reaches the "max orders" of the shipping provider
                    $this->submitFulfillmentsToShippingProviders();
                });
            });
            // submit remaining fulfillments
            $this->submitFulfillmentsToShippingProviders(false);
        } catch (\Throwable $exception) {
            $starshipitFulfillments = collect($this->fulfillments[SalesOrderFulfillment::TYPE_STARSHIPIT]);
            if ($starshipitFulfillments->isNotEmpty()) {
                $starshipitFulfillments->map(function (SalesOrderFulfillment $fulfillment) use ($exception) {
                    Telescope::stopRecording();

                    $this->deleteFulfillment($fulfillment);
                    $this->response[$fulfillment->sales_order_id][$fulfillment->warehouse_id] = Response::getError($exception->getMessage(), Response::CODE_UNEXPECTED, 'id');
                    $this->errorsCount++;
                });
            }
        }

        return $this->response;
    }

    public function count(): int
    {
        if (! isset($this->count)) {
            $this->count = $this->getBuilder()->count();
        }

        return $this->count;
    }

    public function maxSycCountExceeded(): bool
    {
        return $this->count() > self::MAX_SYNC_COUNT;
    }

    public function getErrorsCount(): int
    {
        return $this->errorsCount;
    }

    private function getBuilder()
    {
        $builder = SalesOrder::with([
            'salesOrderLines.warehouse',
            'salesOrderFulfillments',
            'purchaseOrders',
        ]);
        if ($this->request->has('ids')) {
            $builder->whereIn('id', array_unique($this->request->input('ids', [])));
        } else {
            $builder->filter($this->request)->archived($this->request->input('archived', 0));
        }

        return $builder;
    }

    /**
     * How Frontend auto-select fulfillment data (fulfillment_type and shipping_service)
     * i'm looking at warehouse.order_fulfillment
     * if it has 'shipstation' or 'starshipit' then i select one of them. else I select 'manual'
     * warehouse.order_fulfillment i take from GET api/warehouses.
     *
     * for shipping method:
     * if manual entry is added to the order then we set manual entry (requested_shipping_method)
     * else we set  shipping method which the order has (shipping_method_id)
     * else we do not set initial shipping method (edited)
     *
     * yes but it is not likely. because we initialize shipping method when we create an order. let me find what we select there as a default value
     *
     * when we create a new order we use this shipping method as default
     * sales_order_default_shipping_method_domestic
     * from
     * GET api/settings
     *
     * @param  Collection|SalesOrderLine[]  $salesOrderLines
     * @return FulfillSalesOrderRequest|bool true: Already fulfilled
     */
    private function getFulfillmentSalesOrderRequest(SalesOrder $salesOrder, Collection $salesOrderLines, Carbon|string $fulfilledAt)
    {
        /* @var Warehouse $warehouse */
        $warehouse = $salesOrderLines->first()->warehouse;

        $fulfillRequestData = ['warehouse_id' => $warehouse->id, 'fulfilled_at' => $fulfilledAt];

        $fulfillmentType = $warehouse->order_fulfillment ?: SalesOrderFulfillment::TYPE_MANUAL;
        $fulfillRequestData['fulfillment_type'] = $fulfillmentType;

        if ($fulfillmentType == SalesOrderFulfillment::TYPE_MANUAL) {
            $fulfillRequestData['fulfilled_shipping_method_id'] = $salesOrder->shipping_method_id;
            $fulfillRequestData['fulfilled_shipping_method'] = $salesOrder->requested_shipping_method;
        } else {
            $fulfillRequestData['requested_shipping_method_id'] = $salesOrder->shipping_method_id;
            $fulfillRequestData['requested_shipping_method'] = $salesOrder->requested_shipping_method;
        }

        // add lines that not fulfilled yet
        $fulfillRequestData['fulfillment_lines'] = $salesOrderLines->map(function (SalesOrderLine $salesOrderLine) {
            return ['sales_order_line_id' => $salesOrderLine->id, 'quantity' => $salesOrderLine->fulfillable_quantity];
        })->where('quantity', '>', 0)->values()->toArray();

        // all lines already fulfilled or unfulfillable
        if (empty($fulfillRequestData['fulfillment_lines'])) {
            return true;
        }

        return FulfillSalesOrderRequest::createFromCustom($fulfillRequestData, 'POST', ['sales_order' => $salesOrder]);
    }

    /**
     * @param  Collection|SalesOrderLine[]  $salesOrderLines
     * @return DropshipSalesOrderRequest|bool
     */
    private function getDropshipSalesOrderRequest(SalesOrder $salesOrder, Collection $salesOrderLines)
    {
        /* @var Warehouse $warehouse */
        $warehouse = $salesOrderLines->first()->warehouse;

        // already requested
        if ($salesOrder->purchaseOrders->firstWhere('supplier_warehouse_id', $warehouse->id)) {
            return true;
        }

        $dropshipRequestData = ['warehouse_id' => $warehouse->id];
        // shipping methods
        $dropshipRequestData['requested_shipping_method_id'] = $salesOrder->shipping_method_id;
        $dropshipRequestData['requested_shipping_method'] = $salesOrder->requested_shipping_method;
        // lines
        $dropshipRequestData['sales_order_lines'] = $salesOrderLines->map->only('id')->toArray();

        return DropshipSalesOrderRequest::createFromCustom($dropshipRequestData, 'POST', ['sales_order' => $salesOrder]);
    }

    /**
     * Submit fulfillments to the shipping providers
     */
    private function submitFulfillmentsToShippingProviders(bool $satisfyMaxOrdersCount = true)
    {
        $this->submitFulfillmentsToShipstation($satisfyMaxOrdersCount);
        $this->submitFulfillmentsToStarshipit($satisfyMaxOrdersCount);
    }

    /**
     * Submit shipstation fulfillments
     */
    private function submitFulfillmentsToShipstation(bool $satisfyMaxOrdersCount = true)
    {
        $shipstationFulfillments = $this->fulfillments[SalesOrderFulfillment::TYPE_SHIPSTATION];
        if (empty($shipstationFulfillments)) {
            return;
        }

        if (! $satisfyMaxOrdersCount || count($shipstationFulfillments) === self::MAX_SHIPSTATION_ORDERS_COUNT) {
            /** @see https://www.shipstation.com/docs/api/orders/create-update-multiple-orders */
            // TODO:
            // - send to shipstation
            // - check the response then delete failed fulfillments
            // - clear the shipstation fulfillments
        }
    }

    /**
     * Submit starshipit fulfillments
     */
    private function submitFulfillmentsToStarshipit(bool $satisfyMaxOrdersCount = true)
    {
        $starshipitFulfillments = collect($this->fulfillments[SalesOrderFulfillment::TYPE_STARSHIPIT]);
        if (empty($starshipitFulfillments)) {
            return;
        }

        $starshipitInstance = IntegrationInstance::with([])->starshipit()->first();
        if (! $starshipitInstance) {
            $starshipitFulfillments->map(function (SalesOrderFulfillment $fulfillment) {
                $fulfillment->delete();
                $this->response[$fulfillment->sales_order_id][$fulfillment->warehouse_id] = Response::getError(__('messages.integration_instance.not_integrated', ['integration' => 'Starshipit']), Response::CODE_NOT_INTEGRATED, 'id');
                $this->errorsCount++;
            });
        }

        if ($satisfyMaxOrdersCount && $starshipitFulfillments->count() < self::MAX_STARSHIPIT_ORDERS_COUNT) {
            return;
        }

        try {
            // validate the fulfillments
            $filteredFulfillments = $starshipitFulfillments
                ->filter(function (SalesOrderFulfillment $fulfillment) {
                    // check the starshipit order data validated
                    $validationErrors = $fulfillment->toStarshipitOrder()->validate();
                    if (! empty($validationErrors)) {
                        $this->response[$fulfillment->sales_order_id][$fulfillment->warehouse_id] = Response::getError(__('messages.integration_instance.can_not_submit_order', ['resource' => 'Starshipit']), Response::CODE_INTERNAL_VALIDATION, '', $validationErrors);
                        $this->deleteFulfillment($fulfillment);
                        $this->errorsCount++;

                        return false;
                    }

                    // check if the order is already submitted to starshipit
                    $starshipitOrder = StarshipitOrder::with([])->firstOrNew(['sku_fulfillment_id' => $fulfillment->id]);

                    if ($starshipitOrder->exists && $starshipitOrder->order_id) {
                        $this->response[$fulfillment->sales_order_id][$fulfillment->warehouse_id] = Response::getError(__('messages.integration_instance.can_not_submit_order', ['resource' => 'Starshipit']), Response::CODE_INTEGRATION_API_ERROR, 'id', ['AlreadyExists']);
                        $this->deleteFulfillment($fulfillment);
                        $this->errorsCount++;

                        return false;
                    }

                    return true;
                })->values();

            // to starshipit orders
            $starshipitOrders = $filteredFulfillments->map(function (SalesOrderFulfillment $fulfillment) {
                return $fulfillment->toStarshipitOrder();
            })->all();

            $starshipit = new Starshipit($starshipitInstance);
            try {
                // submit to starshipit
                $submitResponse = $starshipit->submitOrders($starshipitOrders);

                // Too Many Requests Error
                if ($submitResponse->statusCode == 429) {
                    sleep(2);
                    $submitResponse = $starshipit->submitOrders($starshipitOrders);
                }

                if (! $submitResponse->body['success']) {
                    // trying to handle errors
                    $submitResponse = $this->linkFulfillmentsWithSSIOrders($filteredFulfillments, $starshipitInstance, $submitResponse);
                }
            } catch (\Throwable $exception) {
                Log::channel('starshipit')->debug('submitting exception', ['message' => $exception->getMessage(), 'code' => $exception->getCode()]);

                $this->cannotSubmitToShippingProvider($filteredFulfillments, $exception->getMessage());
                // clear the starshipit fulfillments
                $this->fulfillments[SalesOrderFulfillment::TYPE_STARSHIPIT] = [];

                return;
            }

            $this->handleStarshipitResponse($submitResponse, $filteredFulfillments);

            // clear the starshipit fulfillments
            $this->fulfillments[SalesOrderFulfillment::TYPE_STARSHIPIT] = [];
        } catch (\Throwable $exception) {
            // try to handle the Starshipit response again without recording the Telescope records
            if (isset($submitResponse) && $submitResponse->statusCode == 200) {
                // stop telescope to handle "too many connections"
                Telescope::stopRecording();

                $this->handleStarshipitResponse($submitResponse, $filteredFulfillments);

                // clear the starshipit fulfillments
                $this->fulfillments[SalesOrderFulfillment::TYPE_STARSHIPIT] = [];

                return;
            }

            throw $exception;
        }
    }

    private function cannotSubmitToShippingProvider(Collection $fulfillments, $error)
    {
        $fulfillments->map(function (SalesOrderFulfillment $fulfillment) use ($error) {
            $this->deleteFulfillment($fulfillment);
            $this->response[$fulfillment->sales_order_id][$fulfillment->warehouse_id] = Response::getError(__('messages.integration_instance.can_not_submit_order', ['resource' => $fulfillment->fulfillment_type]), Response::CODE_INTEGRATION_API_ERROR, 'id', (array) $error);
            $this->errorsCount++;
        });
    }

    private function handleStarshipitResponse(StarshipitResponse $submitResponse, Collection $filteredFulfillments)
    {
        // TODO: sometimes starshipit submit orders partially
        if ($submitResponse->statusCode == 200 && ! ($submitResponse->body['success'] ?? true) && ! empty($submitResponse->body['orders'])) {
            Log::channel('starshipit')->debug('orders submitted partially', $submitResponse->body);

            // TODO: a temporary solution until starshipit fix the issue
            $this->insertStarshipitOrders($submitResponse->body['orders']);
            // only delete the fulfillments that can't submit
            $failedFulfillments = $filteredFulfillments->whereNotIn('fulfillment_number', array_column($submitResponse->body['orders'], 'order_number'))->values();
            $this->cannotSubmitToShippingProvider($failedFulfillments, $submitResponse->body['errors'] ?? $submitResponse->body);
        }

        // all orders submitted successfully
        elseif ($submitResponse->statusCode == 200 && ($submitResponse->body['success'] ?? true)) {
            $this->insertStarshipitOrders($submitResponse->body['orders']);
        } else {
            // failed to submit all orders
            $this->cannotSubmitToShippingProvider($filteredFulfillments, $submitResponse->body['errors'] ?? $submitResponse->body);
        }
    }

    private function insertStarshipitOrders(array $orders)
    {
        $referencePrefix = config('shipstation.order_prefix');
        foreach ($orders as $order) {
            // extract fulfillment id from order's reference
            $skuFulfillmentId = (int) str_replace($referencePrefix, '', $order['reference']);
            try {
                // save starshipit order
                $starshipitOrder = StarshipitOrder::with([])->firstOrNew(['sku_fulfillment_id' => $skuFulfillmentId]);
                $starshipitOrder->json_object = $order;

                $starshipitOrder->save();
            } catch (\Throwable $exception) {
                Log::debug("submitFulfillmentsToStarshipit, sku_fulfillment_id:{$skuFulfillmentId}: {$exception->getMessage()}", $exception->getTrace());
                try {
                    $starshipitOrder = StarshipitOrder::with([])->firstOrNew(['sku_fulfillment_id' => $skuFulfillmentId]);

                    $starshipitOrder->rawOrder = $order;
                    $starshipitOrder->errors = $exception->getMessage();
                    $starshipitOrder->save();
                } catch (\Throwable $exception) {
                    Log::debug("submitFulfillmentsToStarshipit(saving raw), sku_fulfillment_id:{$skuFulfillmentId}: {$exception->getMessage()}", $exception->getTrace());
                }
            }
        }
    }

    private function deleteFulfillment(SalesOrderFulfillment $fulfillment, int $attempts = 3)
    {
        customlog('SKU-6191', 'BulkFulfillSalesOrderService::449 '.$fulfillment->fulfillment_number.' for '.$fulfillment->salesOrder->sales_order_number.' to ShipStation', [
            'debug' => debug_pretty_string(),
        ], 7);
        // try to handle "too many connections" exception, try 3 times
        foreach (range(1, $attempts) as $attempt) {
            try {
                $fulfillment->delete();
                break;
            } catch (\Throwable $exception) {
                // last attempt
                if ($attempt == $attempts) {
                    throw $exception;
                }
                usleep(100);
            }
        }
    }

    private function linkFulfillmentsWithSSIOrders($fulfillments, IntegrationInstance $starshipitInstance, StarshipitResponse $response): StarshipitResponse
    {
        $this->attempts++;

        $starshipit = new Starshipit($starshipitInstance);
        try {
            if (empty($response->body['errors'])) {
                return $response;
            }

            /** @var SalesOrderFulfillment $fulfillment */
            foreach ($fulfillments as $fulfillment) {
                if ($this->hasErrors($response->body['errors'], $fulfillment->fulfillment_number)) {
                    // get starshipit order
                    $getOrderResponse = $starshipit->getOrder($fulfillment->fulfillment_number, 'order_number');
                    if ($getOrderResponse->statusCode == 429) {
                        sleep(2);
                        $getOrderResponse = $starshipit->getOrder($fulfillment->fulfillment_number, 'order_number');
                    }

                    $isArchived = false;
                    if ($getOrderResponse->statusCode == 200 && ($getOrderResponse->body['success'] ?? true)) {
                        $ssiOrder = $getOrderResponse->body['order'];
                        // check if it's archived
                        // unshipped and archived
                        if ($ssiOrder['archived']) {
                            $isArchived = true;
                        } // if the order status is "Printed/Shipped", the Starshipit always return archived=false even though it was archived
                        elseif ($ssiOrder['status'] != 'Unshipped') {
                            // so, search on printed/shipping orders to check if it's not archived
                            $searchResponse = $starshipit->searchOrders(['phrase' => $fulfillment->fulfillment_number])->current();
                            if (empty($searchResponse->body['orders'])) {
                                $isArchived = true;
                            }
                        }
                        if ($isArchived) {
                            $fulfillment->fulfillment_sequence = $fulfillment->fulfillment_sequence + 1;
                            $fulfillment->save();
                            $fulfillment->hasError = false;
                        } else {
                            // compare with lines
                            $fulfillmentToSSIOrder = $fulfillment->toStarshipitOrder();
                            if (count($ssiOrder['items']) == count($fulfillmentToSSIOrder->items)) {
                                // the Starshipit order exists with the same order number, the same lines, and it's not archived
                                StarshipitOrder::with([])->updateOrCreate(['sku_fulfillment_id' => $fulfillment->id], $ssiOrder);
                                if (($ssiOrder['manifested'] ?? false)) {
                                    // mark as fulfilled and add the tracking info
                                    (new GetTrackingJob($starshipitInstance, $fulfillment->id, false))->handle();
                                }
                            } else {
                                // the Starshipit order exists with the same order number, and it's not archived, but with different lines
                                $this->response[$fulfillment->sales_order_id][$fulfillment->warehouse_id] = Response::getError(__('messages.integration_instance.unfulfillable_order', ['resource' => 'Starshipit']), Response::CODE_UNFULFILLABLE_TO_SHIPPING_PROVIDER, 'unfulfillable', ['starshipit_order' => $ssiOrder, 'fulfillment_sequence' => $fulfillment->fulfillment_sequence]);
                                $this->deleteFulfillment($fulfillment);
                                $this->errorsCount++;
                            }
                            $fulfillment->hasError = true;
                        }
                    } else {
                        $fulfillment->hasError = true;
                    }
                } else {
                    $fulfillment->hasError = false;
                }
            }

            // send fulfillments that hasError=false again
            if ($fulfillments->where('hasError', false)->isNotEmpty()) {
                $starshipitOrders = $fulfillments->where('hasError', false)->map(function (SalesOrderFulfillment $fulfillment) {
                    return $fulfillment->toStarshipitOrder();
                })->all();
                $response = $starshipit->submitOrders($starshipitOrders);
                if (! $response->body['success'] && $this->attempts < 5) {
                    $response = $this->linkFulfillmentsWithSSIOrders($fulfillments, $starshipitInstance, $response);
                }
            }

            return $response;
        } catch (\Throwable $exception) {
            return $response;
        }
    }

    private function hasErrors($errors, $id): bool
    {
        foreach ($errors as $error) {
            if (str_contains($error['details'], $id)) {
                return true;
            }
        }

        return false;
    }
}
