<?php

namespace App\Services\SalesOrder;

use App\Helpers;
use App\Http\Requests\DropshipSalesOrderRequest;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\BackorderQueue;
use App\Models\OrderLink;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\SupplierInventory;
use App\Models\Warehouse;
use App\Repositories\ProductRepository;
use App\Services\InventoryManagement\BackorderManager;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\InventoryManagement\InventoryReductionCause;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;

class ApproveSalesOrderService extends SalesOrderService
{
    const WITHOUT_APPROVE_DROPSHIP = 0;

    const APPROVE_DROPSHIP_CREATE_OR_UPDATE = 1;

    const APPROVE_DROPSHIP_UPDATE_ONLY = 2;

    protected BackorderManager $backorderManager;

    private ProductRepository $productRepository;

    /**
     * ApproveSalesOrderService constructor.
     */
    public function __construct(SalesOrder $salesOrder)
    {
        parent::__construct($salesOrder);
        $this->productRepository = app(ProductRepository::class);
        $this->backorderManager = app(BackorderManager::class);
    }

    public function approve(int $approveDropship = 0, bool $onlyForSufficientStock = false, bool $reserve = false): bool
    {
        //    if ( ( $completed = $this->salesOrder->isComplete() ) !== true )
        //    {
        //      return $completed;
        //    }

        if ($this->salesOrder->order_status == SalesOrder::STATUS_CLOSED || ! $this->salesOrder->allProductsMapped()) {
            return false;
        }

        DB::transaction(function () use ($approveDropship, $onlyForSufficientStock, $reserve) {
            // add inventory movements to sales order lines
            ! Model::preventsLazyLoading() && $this->salesOrder->load([
                'salesOrderLines',
                'salesOrderLines.product',
                'salesOrderLines.warehouse',
                'salesOrderLines.inventoryMovements',
            ]);

            // approve sales order lines(adding inventory movements or purchase orders for dropship lines)
            $this->approveNormalLines();

            if ($approveDropship) {
                $this->approveDropshipLines(
                    $approveDropship == self::APPROVE_DROPSHIP_UPDATE_ONLY,
                    $onlyForSufficientStock
                );
            }

            $this->markAsApproved($reserve);

            $this->syncBackorderQueueCoveragesAfterApproval();

            // Update the fulfillment and other statuses of order
            $this->salesOrder->updateFulfillmentStatus();

            $this->salesOrder->cacheCurrencyRate(true);
        });

        return true;
    }

    private function syncBackorderQueueCoveragesAfterApproval(): void
    {
        $this->salesOrder->salesOrderLines()
            ->whereHas('product', function ($query) {
                $query->where('no_audit_trail', 0);
            })
            ->where('is_product', 1)
            ->whereNotNull('warehouse_id')
            ->each(function (SalesOrderLine $orderLine) {
                dispatch(new SyncBackorderQueueCoveragesJob(null, null, Arr::wrap($orderLine->product_id), $orderLine->warehouse_id));
            });
    }

    public function reserve(int $approveDropship = 0, $onlyForSufficientStock = false)
    {
        $this->approve($approveDropship, $onlyForSufficientStock, true);
        $this->salesOrder->order_status = SalesOrder::STATUS_RESERVED;
        $this->salesOrder->update();

        return true;
    }

    /**
     * Approve Dropship Request.
     *
     * Since this is a dropship order, no inventory movements are needed and no fifo layers are affected
     * Instead, a purchase order is created
     */
    public function approveDropshipRequest(DropshipSalesOrderRequest $request): PurchaseOrder
    {
        return $this->processDropship($request->validated());
    }

    public function processDropship(array $inputs): PurchaseOrder
    {
        // Set shipping method id if present
        //    if(isset($inputs['shipping_method_id'])){
        //      $this->salesOrder->shipping_method_id = $inputs['shipping_method_id'];
        //      $this->salesOrder->save();
        //    }

        $warehouse = Warehouse::with([])->findOrFail($inputs['warehouse_id']);

        // create a new purchase order
        $purchaseOrder = new PurchaseOrder();
        $purchaseOrder->purchase_order_date = $this->salesOrder->order_date;
        $purchaseOrder->requested_shipping_method_id = $inputs['requested_shipping_method_id'] ?? $inputs['shipping_method_id'] ?? null;
        $purchaseOrder->requested_shipping_method = $inputs['requested_shipping_method'] ?? null;
        if (empty($purchaseOrder->requested_shipping_method_id) && empty($purchaseOrder->requested_shipping_method)) {
            $purchaseOrder->requested_shipping_method_id = $this->salesOrder->shipping_method_id;
            $purchaseOrder->requested_shipping_method = $this->salesOrder->requested_shipping_method;
        }
        $purchaseOrder->receipt_status = PurchaseOrder::RECEIPT_STATUS_DROPSHIP;
        $purchaseOrder->supplier_id = $warehouse->supplier_id;
        $purchaseOrder->supplier_warehouse_id = $inputs['warehouse_id'];
        $purchaseOrder->destination_address_id = $this->salesOrder->shipping_address_id;
        $purchaseOrder->currency_id = $this->salesOrder->currency_id;
        $purchaseOrder->supplier_notes = $inputs['supplier_notes'] ?? null;
        $this->salesOrder->purchaseOrders()->save($purchaseOrder);

        // create link between SO and PO
        $orderLink = new OrderLink();
        $orderLink->child = $purchaseOrder;
        $orderLink->link_type = OrderLink::LINK_TYPE_DROPSHIP;
        $this->salesOrder->childLinks()->save($orderLink);

        // add sales order lines to purchase order lines
        $salesOrderLines = SalesOrderLine::with([])->whereIn('id',
            array_column($inputs['sales_order_lines'], 'id'))->get();
        $purchaseOrderLines = $salesOrderLines->map(function (SalesOrderLine $line) use ($purchaseOrder) {
            $amount = $this->productRepository
                ->getCostForSupplier($line->product, $line->warehouse_id,
                    $purchaseOrder->supplier_id);
            $supplierId = $line->warehouse->supplier_id;

            return $line->only(['description', 'product_id', 'quantity']) +
                   ['amount' => $amount];
        });
        $purchaseOrder->setPurchaseOrderLines($purchaseOrderLines->toArray());

        // submit to supplier
        $purchaseOrder->submit(true, true);
        // set sales order's fulfillment status as awaiting_tracking
        $this->salesOrder->updateFulfillmentStatus();

        return $purchaseOrder;
    }

    public function markAsApproved(bool $reserve = false, ?Carbon $approvedDate = null)
    {
        if ($this->salesOrder->order_status == SalesOrder::STATUS_CLOSED) {
            return;
        }

        // change order status to Open and set approved date to now
        $this->salesOrder->order_status = $reserve ? SalesOrder::STATUS_RESERVED : SalesOrder::STATUS_OPEN;
        $this->salesOrder->approved_at = ($reserve ? $this->salesOrder->reserved_at : $this->salesOrder->approved_at) ?: ($approvedDate ?: now());
        $this->salesOrder->save();

        // reprint the packing slips
        //        AddPackingSlipQueueObserver::salesOrderUpdated($this->salesOrder);
    }

    /**
     * Approve Dropship Sales order Lines.
     *
     * Since this is a dropship order, no inventory movements are needed and no fifo layers are affected
     * Instead, a purchase order is created
     */
    private function approveDropshipLines(bool $updateOnly = false, bool $onlyForSufficientStock = false)
    {
        $dropshipLines = $this->salesOrder->salesOrderLines->where('is_dropship', true)->groupBy('warehouse_id');

        $purchaseOrderIds = []; // for syncing droipship purchase orders

        /** @var Collection|SalesOrderLine[] $lines */
        foreach ($dropshipLines as $warehouseId => $lines) {
            if ($onlyForSufficientStock) {
                $lines = $lines->filter(function (SalesOrderLine $line) {
                    if (! $line->is_product || ! $line->warehouse->supplier->auto_fulfill_dropship) {
                        return false;
                    }
                    /** @var SupplierInventory $inventory */
                    $inventory = $line->warehouse->supplierInventory()
                        ->where('product_id', $line->product_id)
                        ->first();

                    return $inventory &&
                 (($inventory->quantity && $inventory->quantity >= $line->quantity) ||
                   $inventory->in_stock);
                });
            }

            if ($lines->isEmpty()) {
                continue;
            }

            $supplierId = $lines->first()->warehouse->supplier_id;

            // get/create a new purchase order
            $purchaseOrder = $this->salesOrder->purchaseOrders()->where('supplier_warehouse_id', $warehouseId)->first();
            if (! $purchaseOrder) {
                $purchaseOrder = new PurchaseOrder();
            }

            if ($updateOnly && ! $purchaseOrder->exists) {
                continue;
            }

            // At this stage, if the purchase order is Submission Status: Finalized, no line items can be removed or modified
            if ($purchaseOrder->submission_status == PurchaseOrder::SUBMISSION_STATUS_FINALIZED) {
                $purchaseOrderIds[] = $purchaseOrder->id; // add to dropship purchase orders

                continue;
            }

            $purchaseOrder->purchase_order_date = $this->salesOrder->order_date;
            $purchaseOrder->requested_shipping_method_id = $this->salesOrder->shipping_method_id;
            $purchaseOrder->submission_format = $purchaseOrder->submission_format ?: PurchaseOrder::SUBMISSION_FORMAT_EMAIL_PDF_AND_CSV_ATTACHMENTS;
            $purchaseOrder->receipt_status = PurchaseOrder::RECEIPT_STATUS_DROPSHIP;
            $purchaseOrder->supplier_id = $supplierId;
            $purchaseOrder->supplier_warehouse_id = $warehouseId;
            $purchaseOrder->destination_address_id = $this->salesOrder->shipping_address_id;
            $purchaseOrder->currency_id = $this->salesOrder->currency_id;
            $this->salesOrder->purchaseOrders()->save($purchaseOrder);

            // create link between SO and PO
            if ($purchaseOrder->wasRecentlyCreated) {
                $orderLink = new OrderLink();
                $orderLink->child = $purchaseOrder;
                $orderLink->link_type = OrderLink::LINK_TYPE_DROPSHIP;
                $this->salesOrder->childLinks()->save($orderLink);
            }

            $purchaseOrderLines = $lines->map(function (SalesOrderLine $line) use ($purchaseOrder) {
                $existedPOL = $purchaseOrder->purchaseOrderLines->firstWhere('product_id', $line->product_id);

                $amount = $this->productRepository
                    ->getCostForSupplier($line->product, $line->warehouse_id,
                        $purchaseOrder->supplier_id);

                return $line->only(['description', 'product_id', 'quantity']) +
                       ['id' => $existedPOL?->id, 'amount' => $amount];
            });

            $purchaseOrder->setPurchaseOrderLines($purchaseOrderLines->toArray(), false);
            $purchaseOrder->load('purchaseOrderLines');

            // Approve the PO
            $purchaseOrder->approve();
            // submit to supplier
            $purchaseOrder->submit(true, true);
            // set sales order's fulfillment status as awaiting_tracking
            $this->salesOrder->updateFulfillmentStatus();

            $purchaseOrderIds[] = $purchaseOrder->id; // add to dropship purchase orders
        }

        // sync dropship purchase orders
        $this->salesOrder->purchaseOrders()->whereNotIn('id', $purchaseOrderIds)->delete();
    }

    /**
     * Approve Normal(not dropship and product) Sales order lines.
     *
     * @throws \Exception
     */
    private function approveNormalLines()
    {
        $salesChannel = $this->salesOrder->salesChannel;

        if ($salesChannel->integrationInstance->isShopify()) {
            if (! $this->salesOrder->shopifyOrder) {
                throw new InvalidArgumentException("Shopify order missing for sales order: {$this->salesOrder->sales_order_number}");
            }
            if ($is_pre_audit_trail = $this->salesOrder->shopifyOrder->isPreAuditTrail()) {
                if ($this->salesOrder->shopifyOrder::getOrderStatus($this->salesOrder->shopifyOrder->toArray()) == 'closed') {
                    $this->salesOrder->close();
                    $this->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_FULFILLED;
                } else {
                    if (collect($this->salesOrder->shopifyOrder->toArray()['fulfillments'])->where('status', '!=', 'cancelled')->count()) {
                        $this->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_PARTIALLY_FULFILLED;
                    }
                }
                $this->salesOrder->save();
            }
        } else {
            $is_pre_audit_trail = false;
        }

        if ($this->salesOrder->order_date->lt(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE), 'UTC'))) {
            $is_pre_audit_trail = true;
            if ($this->salesOrder->order_status == SalesOrder::STATUS_CLOSED) {
                $this->salesOrder->salesOrderLines()->update(['no_audit_trail' => true]);
            }
        }

        if (! $this->salesOrder->salesOrderLines->where('is_product', true)->count()) {
            $this->salesOrder->close();
            $this->salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_FULFILLED;
        }

        /** @var SalesOrderLine $salesOrderLine */
        foreach ($this->salesOrder->salesOrderLines as $salesOrderLine) {
            if ($salesOrderLine->warehouse && $salesOrderLine->warehouse->type == Warehouse::TYPE_AMAZON_FBA) {
                continue;
            //        $this->approveFBALine($salesOrderLine);
            } // if line is product and linked with system
            elseif ($salesOrderLine->is_product == false) {
                continue;
            } elseif ($salesOrderLine->product) {
                $product = $salesOrderLine->product;

                if ($salesOrderLine->warehouse_id) {
                    if ($salesChannel->integrationInstance->isShopify() && $is_pre_audit_trail) {
                        $tally_fulfilled = 0;
                        foreach ($this->salesOrder->shopifyOrder->toArray()['fulfillments'] as $fulfillment) {
                            if ($fulfillment['status'] == 'cancelled') {
                                continue;
                            }
                            foreach ($fulfillment['line_items'] as $line_item) {
                                if ($line_item['id'] == $salesOrderLine->sales_channel_line_id) {
                                    if ($line_item['fulfillment_status'] == 'partial') {
                                        $newSalesOrderLine = new SalesOrderLine();

                                        $newSalesOrderLine->no_audit_trail = true;
                                        $newSalesOrderLine->is_product = true;
                                        $newSalesOrderLine->quantity = $line_item['quantity'];
                                        $newSalesOrderLine->setCogsIfFinancialLineExists($product->getUnitCostAtWarehouse($salesOrderLine->warehouse_id));
                                        $newSalesOrderLine->sales_channel_line_id = $salesOrderLine->sales_channel_line_id;
                                        $newSalesOrderLine->product_id = $salesOrderLine->product_id;
                                        $newSalesOrderLine->description = $salesOrderLine->description;
                                        $newSalesOrderLine->warehouse_id = $salesOrderLine->warehouse_id;
                                        $newSalesOrderLine->sales_order_id = $salesOrderLine->sales_order_id;
                                        $newSalesOrderLine->split_from_line_id = $salesOrderLine->id;
                                        $newSalesOrderLine->save();

                                        $salesOrderLine->quantity = $line_item['fulfillable_quantity'];
                                        $salesOrderLine->save();
                                    } else {
                                        $tally_fulfilled += $line_item['quantity'];
                                    }
                                }
                            }
                        }
                        if ($tally_fulfilled) {
                            $salesOrderLine->no_audit_trail = true;
                        } else {
                            $salesOrderLine->no_audit_trail = false;
                        }
                        //dd(collect($this->salesOrder->shopifyOrder->toArray()['fulfillments']['line_items'])->flatten()->where('id', $salesOrderLine->sales_channel_line_id)->sum('quantity'));
                    }
                }

                if (! $salesOrderLine->no_audit_trail && $salesOrderLine->warehouse_id && $salesOrderLine->product) {
                    if (! $salesOrderLine->is_dropship && $salesOrderLine->quantity > 0) {
                        /**
                         * We simply reduce inventory by the quantity of the sales order line
                         * and create movements for each of the layers involved.
                         */
                        $manager = new InventoryManager($salesOrderLine->warehouse_id, $salesOrderLine->product);
                        /** @see SKU-4440 for blemished products only use fifo layers */
                        $useOnlyFifoLayers = $salesOrderLine->product->type == Product::TYPE_BLEMISHED;

                        if ($salesOrderLine->inventoryMovements()->count()) {
                            /**
                             * Existing sales order lines must have inventory movements.
                             * Note that $salesOrderLine->wasRecentlyCreated == false
                             * does not work (for some reason), perhaps because the
                             * sales order lines are loaded for the approval and the
                             * field doesn't get updated.
                             */
                            $quantity = abs(collect($salesOrderLine->getReductionActiveMovements()->toArray())->sum('quantity'));
                            if ($quantity > $salesOrderLine->quantity) {
                                $manager->decreaseNegativeEventQty($quantity - $salesOrderLine->quantity, $salesOrderLine);
                            } elseif ($quantity < $salesOrderLine->quantity) {
                                $manager->increaseNegativeEventQty($salesOrderLine->quantity - $quantity, $salesOrderLine, $useOnlyFifoLayers, $salesOrderLine);
                            }
                        } else {
                            if ($is_pre_audit_trail == false) {
                                $manager->takeFromStock($salesOrderLine->quantity, $salesOrderLine, $useOnlyFifoLayers, $salesOrderLine);
                            }
                        }

                        $salesOrderLine->load('backorderQueue');
                    }
                } else {
                    // reset any previous movements
                    $salesOrderLine->resetMovements();
                }

                // set unit cost, other changes and cache has_backorder
                $salesOrderLine->save();
            }
        }
    }

    protected function createBackorderQueue(int $quantity, SalesOrderLine $salesOrderLine): BackorderQueue
    {
        $defaultSupplierProduct = $salesOrderLine->product->defaultSupplierProduct;

        return $this->backorderManager->createBackorderQueue(
            $quantity,
            $salesOrderLine->id,
            InventoryReductionCause::INCREASED_NEGATIVE_EVENT,
            $defaultSupplierProduct?->supplier_id
        );
    }
}
