<?php

namespace App\Services\SalesOrder\Actions;

use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Warehouse;
use App\Repositories\WarehouseRepository;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use Throwable;

class SetLineWarehouse
{
    public function __construct(private readonly WarehouseRepository $warehouses)
    {
    }

    /**
     * @throws InvalidProductWarehouseRouting
     * @throws Throwable
     */
    public function setWarehouse(SalesOrderLine $orderLine): SalesOrderLine
    {
        // Externally fulfilled lines don't have a warehouse or movements
        if ($orderLine->externally_fulfilled_quantity > 0 && $orderLine->externally_fulfilled_quantity == $orderLine->quantity) {
            $orderLine->warehouse_id = null;

            return $orderLine;
        }

        if ($this->isWarehouseRouting($orderLine)) {
            // Routing type warehouse must already have a warehouse specified.
            throw_if(is_null($orderLine->warehouse_id),
                'Sales order line: '.$orderLine->id.' has warehouse routing method without a warehouse');

            return $orderLine;
        }

        if ($this->isDropshipRouting($orderLine)) {
            return $this->setDropshipWarehouse($orderLine);
        }

        // At this point, if the line already has an assigned warehouse,
        // we abort and keep that warehouse..
        if (! empty($orderLine->warehouse_id)) {
            return $orderLine;
        }

        // At this point, we advanced routing warehouse selection method or

        $warehouseId = $this->warehouses->getPriorityStockWarehouseIdForProduct($orderLine->product,
            $orderLine->processableQuantity, false, $orderLine->salesOrder->shippingAddress ?? null);
        $warehousesWithStock = collect($this->warehouses->getWarehousesWithInventoryForProduct($orderLine->product,
            $orderLine->salesOrder->shippingAddress ?? null))
            ->sortByDesc('quantity')
            ->unique('warehouse_id')
            ->values();

        if (Setting::getValueByKey(Setting::KEY_AUTO_SPLIT_SALES_ORDER_LINE_ACROSS_WAREHOUSES) && is_null($warehouseId) && count($warehousesWithStock) > 0) {
            $orderLine->splitSalesOrderLineByWarehouses($warehousesWithStock);
        } else {
            return $orderLine->setWarehouseId($this->warehouses
                ->getPriorityStockWarehouseIdForProduct(
                    product       : $orderLine->product,
                    quantityNeeded: $orderLine->processableQuantity,
                    orderAddress  : $orderLine->salesOrder->shippingAddress ?? null
                ));
        }

        return $orderLine;
    }

    /**
     * @throws InvalidProductWarehouseRouting
     */
    private function setDropshipWarehouse(SalesOrderLine $orderLine): SalesOrderLine
    {
        // The line must be dropshippable and must be linked to a dropship enabled warehouse,
        // or we select a dropship enabled warehouse that's applicable.
        if (! $orderLine->product->is_dropshippable || ($orderLine->warehouse && ! $orderLine->warehouse->dropship_enabled)) {
            // Dropship is requested for a line that cannot be dropshipped or
            // the selected warehouse doesn't have dropship enabled. Either way,
            // this is a user error, and we abort the warehouse selection.
            throw new InvalidProductWarehouseRouting(
                product: $orderLine->product,
                message: 'SKU: '.$orderLine->product->sku.' has dropship warehouse routing '.(! $orderLine->product->is_dropshippable ?
                ' but is not dropshippable.' : ' but the selected warehouse is not dropship enabled.')
            );
        } elseif (! $orderLine->warehouse) {
            // We select a dropship warehouse for the line.
            $orderLine->setWarehouseId(
                $this->warehouses->getDropshippableWarehouseForProduct($orderLine->product)
            );
        }

        return $orderLine;
    }

    private function isWarehouseRouting(SalesOrderLine $orderLine): bool
    {
        return $orderLine->warehouse_routing_method == WarehouseRoutingMethod::WAREHOUSE;
    }

    private function isDropshipRouting(SalesOrderLine $orderLine): bool
    {
        return $orderLine->warehouse_routing_method == WarehouseRoutingMethod::DROPSHIP;
    }
}
