<?php

namespace App\Services\SalesOrder\Actions;

use App\Data\SalesOrderLineData;
use App\Data\UpdateSalesOrderData;
use App\DTO\ProductWarehousePairDto;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Services\InventoryManagement\InventoryManager;
use Closure;
use Exception;
use Illuminate\Support\Facades\DB;
use Spatie\LaravelData\Optional;
use Throwable;

class HandleSalesOrderLineCancellations
{
    /**
     * @throws Exception
     */
    public function handle(UpdateSalesOrderData $data, Closure $next)
    {
        if ($data->payload->sales_order_lines instanceof Optional) {
            return $next($data);
        }

        $cancellableLines = $data->payload->sales_order_lines->filter(fn(SalesOrderLineData $line) => !$line->quantity_to_cancel instanceof Optional && $line->quantity_to_cancel > 0);

        if ($cancellableLines->count() == 0) {
            return $next($data);
        }

        $data->payload->shouldDeleteFulfillments = true;
        $data->payload->deleteFulfillmentsReason = 'Sales order lines cancelled (' . $cancellableLines->map(fn(SalesOrderLineData $line) => $line->salesOrderLine?->sku ?? $line->description)->toCollection()->implode(', ') . ')';

        $data->payload->sales_order_lines->each(function (SalesOrderLineData $line) use (&$data) {
            if ($line->quantity_to_cancel instanceof Optional || $line->salesOrderLine instanceof Optional) {
                return $line;
            }

            if ($line->quantity_to_cancel == 0 && $line->salesOrderLine?->canceled_quantity > 0) {
                $line->canceled_quantity = 0;
                return $line;
            }

            if ($line->quantity_to_cancel > 0) {
                return $this->updateCanceledQuantity($data, $line);
            }

            return $line;
        });

        return $next($data);
    }

    /**
     * @throws Throwable
     */
    private function updateCanceledQuantity(UpdateSalesOrderData $data, SalesOrderLineData $line): SalesOrderLineData
    {
        $existingSalesOrderLine = $line->salesOrderLine;

        $qtyNewlyCanceled = max($line->quantity_to_cancel - $existingSalesOrderLine->canceled_quantity, 0);
        if($qtyNewlyCanceled == 0){
            $line->quantity = $line->quantity - $existingSalesOrderLine->canceled_quantity;
            return $line;
        }

        $cancelledQuantity = min($line->quantity_to_cancel, $line->quantity);
        $updatedQuantity = max(0, $existingSalesOrderLine->quantity - $cancelledQuantity);

        $line->canceled_quantity = $cancelledQuantity;
        $line->quantity = $updatedQuantity;
        customlog('cancelQty', $existingSalesOrderLine->product?->sku.' for '.$existingSalesOrderLine->salesOrder->sales_order_number.' qty: '.$existingSalesOrderLine->quantity.' canceled: '.$cancelledQuantity.' updated: '.$updatedQuantity);
        customlog('cancelQty', 'after save qty: '.$line->quantity.' canceled: '.$line->canceled_quantity);
        if ($existingSalesOrderLine->product && $existingSalesOrderLine->warehouse && $qtyNewlyCanceled > 0) {
            DB::transaction(function () use ($qtyNewlyCanceled, $existingSalesOrderLine) {
                InventoryManager::with($existingSalesOrderLine->warehouse_id, $existingSalesOrderLine->product)
                    ->decreaseNegativeEventQty($qtyNewlyCanceled, $existingSalesOrderLine);
            });
            // We re-calculate backorder queue coverages.
            if ($data->productWarehousePairsNeedingCoverageUpdate instanceof Optional) {
                $data->productWarehousePairsNeedingCoverageUpdate = ProductWarehousePairDto::collection([
                    ProductWarehousePairDto::from([
                        'product_id' => $existingSalesOrderLine->product_id,
                        'warehouse_id' => $existingSalesOrderLine->warehouse_id,
                    ]),
                ]);
            } else {
                $data->productWarehousePairsNeedingCoverageUpdate[] = ProductWarehousePairDto::from([
                    'product_id' => $existingSalesOrderLine->product_id,
                    'warehouse_id' => $existingSalesOrderLine->warehouse_id,
                ]);
            }
        }

        return $line;

    }
}