<?php

namespace App\Services\SalesOrder\Actions;

use App\Data\AddressableData;
use App\Data\UpdateSalesOrderData;
use App\Models\Addressable;
use App\Models\Customer;
use App\Repositories\CustomerRepository;
use Closure;
use Spatie\LaravelData\Optional;

class FindCustomerForSalesOrder
{
    public function __construct(
        private readonly CustomerRepository $customers,
    ) {}

    public function handle(UpdateSalesOrderData $data, Closure $next)
    {
        // Currently we are not allowing changing of customer if it already exists
        if ($data->salesOrder->customer_id || $data->payload->customer instanceof Optional || !$data->payload->customer) {
            return $next($data);
        }

        $customerPayload = $data->payload->customer;

        if ($existingCustomer = Customer::exists($customerPayload->toArray())) {
            $this->upsertAddressableData($data, $existingCustomer);
        } elseif (is_null($customerPayload->name)) {
            return $next($data);
        } else {
            $this->setDefaultAddressIds($data);
            $customerCollection = $this->customers->saveWithRelations($customerPayload);
            $existingCustomer = $customerCollection->first();
        }

        $data->payload->customer_id = $existingCustomer['id'];

        return $next($data);
    }

    private function upsertAddressableData(UpdateSalesOrderData $data, Customer $existingCustomer): void
    {
        $addressableCollection = collect();
        if ($data->payload->shipping_address_id) {
            $addressableCollection->push(AddressableData::from([
                'address_id' => $data->payload->shipping_address_id,
                'addressable_id' => $existingCustomer->id,
                'addressable_type' => Customer::class,
            ]));
        }
        if ($data->payload->billing_address_id) {
            $addressableCollection->push(AddressableData::from([
                'address_id' => $data->payload->billing_address_id,
                'addressable_id' => $existingCustomer->id,
                'addressable_type' => Customer::class,
            ]));
        }
        Addressable::upsert($addressableCollection->toArray(), ['address_id', 'addressable_id', 'addressable_type']);
    }

    private function setDefaultAddressIds(UpdateSalesOrderData $data): void
    {
        if ($data->payload->shipping_address_id) {
            $data->payload->customer->default_shipping_address_id = $data->payload->shipping_address_id;
        }
        if ($data->payload->billing_address_id) {
            $data->payload->customer->default_billing_address_id = $data->payload->billing_address_id;
        }
    }
}