<?php

namespace App\Services\SalesOrder\Actions;

use App\Exceptions\InsufficientStockException;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Services\SalesOrder\SalesOrderLineInventoryManager;
use Illuminate\Support\Collection;
use Throwable;

/**
 * Approves a Sales Order.
 * Note that only a draft order can be approved.
 * Therefore, the Action is only intended to be used
 * for draft sales orders.
 */
class ApproveSalesOrder
{
    public function __construct(private readonly SalesOrderLineInventoryManager $lineInventoryManager)
    {
    }

    /**
     * @throws InsufficientStockException
     */
    public function approve(SalesOrder $order, bool $reserve = false, ?Collection $originalLines = null): SalesOrder
    {
        $order->setStatus(status: $reserve ? SalesOrder::STATUS_RESERVED : SalesOrder::STATUS_OPEN);

        // We handle inventory reservation for the sales order.
        return $this->reserveInventoryForOrder($order, $originalLines);
    }

    /**
     * @throws InsufficientStockException
     * @throws Throwable
     */
    private function reserveInventoryForOrder(SalesOrder $order, ?Collection $originalLines = null): SalesOrder
    {
        $order->warehousedProductLines()->nonFba()->each(function (SalesOrderLine $orderLine) use ($originalLines) {
            $this->lineInventoryManager->refreshLineReservation(
                orderLine: $orderLine,
                originalLine: $originalLines?->where('id', $orderLine->id)->first()
            );
        });

        return $order;
    }
}
