<?php

namespace App\Services\PurchaseOrder;

use App\Collections\PurchaseOrderLineCollection;
use App\Data\CreateDropshipPurchaseOrderData;
use App\DTO\PurchaseOrderDto;
use App\DTO\PurchaseOrderLineDto;
use App\Helpers;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\Currency;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\Warehouse;
use App\Repositories\BackorderQueueRepository;
use App\Repositories\PurchaseOrderRepository;
use Illuminate\Support\Facades\DB;
use Throwable;

class PurchaseOrderManager
{
    private PurchaseOrderRepository $purchaseOrderRepository;

    private BackorderQueueRepository $backorderQueueRepository;

    public function __construct(private readonly Supplier $supplier)
    {
        $this->purchaseOrderRepository = app(PurchaseOrderRepository::class);
        $this->backorderQueueRepository = app(BackorderQueueRepository::class);
    }

    /**
     * @throws Throwable
     */
    public function processBackorderPurchasing(): void
    {
        //Log::debug('Starting process backorder purchasing for supplier ' . $this->supplier->name);
        $productIds = [];
        $purchaseOrderIds = [];

        DB::transaction(function () use (&$productIds, &$purchaseOrderIds) {
            $lines = $this->getBackorderPurchaseOrderBreakdownForSupplier($this->supplier);

            foreach ($lines as $warehouseId => $purchaseOrderLinesData) {
                /** @var Warehouse $warehouse */
                $warehouse = Warehouse::query()->findOrFail($warehouseId);

                $purchaseOrderDto = PurchaseOrderDto::from([
                    'purchase_order_date' => now(),
                    'receipt_status' => PurchaseOrder::RECEIPT_STATUS_UNRECEIVED,
                    'supplier_id' => $this->supplier->id,
                    'destination_warehouse_id' => $warehouse->id,
                    'currency_code' => $this->supplier->defaultPricingTier?->currency_code ?? Currency::default()->code,
                    'submission_format' => $this->supplier->purchase_order_format ?? Helpers::setting(Setting::KEY_PO_FORMAT),
                ]);

                if ($this->supplier->auto_split_backorder_po_by_brand) {
                    foreach ($purchaseOrderLinesData as $brandId => $purchaseOrderLines) {
                        //Log::debug("\tProcessing for warehouse $warehouseId and brandId $brandId");

                        $purchaseOrderDto->purchaseOrderLines = PurchaseOrderLineDto::collection(new PurchaseOrderLineCollection($purchaseOrderLines));
                        $purchaseOrder = $this->createAndApprovePurchaseOrder($purchaseOrderDto);
                        $purchaseOrderIds[] = $purchaseOrder->id;

                        if ($this->supplier->auto_submit_backorder_po) {
                            $this->submitPurchaseOrder($purchaseOrder);
                        }
                    }
                } else {
                    //Log::debug("\tProcessing for warehouse $warehouseId");

                    $purchaseOrderDto->purchaseOrderLines = PurchaseOrderLineDto::collection((new PurchaseOrderLineCollection($purchaseOrderLinesData)));
                    $purchaseOrder = $this->createAndApprovePurchaseOrder($purchaseOrderDto);
                    $purchaseOrderIds[] = $purchaseOrder->id;
                    if ($this->supplier->auto_submit_backorder_po) {
                        $this->submitPurchaseOrder($purchaseOrder);
                    }
                }
                if (! $this->supplier->auto_submit_backorder_po) {
                    customlog('backorderPurchasing', 'Supplier '.$this->supplier->name.' is not set to auto submit backorder purchase orders', [], 7);
                }
            }
        });

        if ($this->supplier->auto_receive_backorder_po) {
            foreach ($purchaseOrderIds as $purchaseOrderId) {
                //Log::debug("\tReceiving PO $purchaseOrderId");
                $this->receivePurchaseOrder($purchaseOrderId);
            }
        } else {
            if (count($purchaseOrderIds) > 0) {
                $purchaseOrderLineIds = PurchaseOrderLine::query()
                    ->whereIn('purchase_order_id', $purchaseOrderIds)
                    ->pluck('id')
                    ->toArray();
                dispatch(new SyncBackorderQueueCoveragesJob($purchaseOrderLineIds));
            }
        }

        //Log::debug('Finished process backorder purchasing for supplier ' . $this->supplier->name);
    }

    public function createAndApprovePurchaseOrder(PurchaseOrderDto $purchaseOrderDto): PurchaseOrder
    {
        $purchaseOrder = $this->createPurchaseOrder($purchaseOrderDto);
        $purchaseOrder->refresh();
        $purchaseOrder->approve(null, false);

        return $purchaseOrder;
    }

    public function createPurchaseOrder(PurchaseOrderDto|CreateDropshipPurchaseOrderData $data): PurchaseOrder
    {
        return $this->purchaseOrderRepository->createPurchaseOrder($data);
    }

    private function submitPurchaseOrder(PurchaseOrder $purchaseOrder): void
    {
        //Log::debug("\tSubmitting PO {$purchaseOrder->id}");
        $purchaseOrder->submit();
        //Log::debug("\tSubmitted PO {$purchaseOrder->id}");
    }

    public function receivePurchaseOrder(int $purchaseOrderId): void
    {
        //Log::debug("\tReceiving PO $purchaseOrderId");
        (new ShipmentManager())->receiveShipment([
            'purchase_order_id' => $purchaseOrderId,
            'received_at' => now(),
        ]);
        //Log::debug("\tReceived PO $purchaseOrderId");
    }

    public function getBackorderPurchaseOrderBreakdownForSupplier(Supplier $supplier): array
    {
        $lines = [];

        $this->backorderQueueRepository->getWarehousesNeedingBackordersForSupplier($supplier)->pluck('warehouse_id')->each(function (int $warehouseId) use ($supplier, &$lines) {
            /** @var Warehouse $warehouse */
            $warehouse = Warehouse::query()->findOrFail($warehouseId);

            $backorders = $this->backorderQueueRepository->getProductQuantityNeededForSupplierAndWarehouse($supplier, $warehouse)->toArray();
            $inbounds = $this->purchaseOrderRepository->getProductQuantityInboundForSupplierAndWarehouse($supplier, $warehouse)->pluck('quantity', 'product_id')->toArray();

            foreach ($backorders as $product) {
                $quantity_needed = $product['quantity'] - ($inbounds[$product['product_id']] ?? 0);
                if ($quantity_needed > 0) {
                    if ($supplier->auto_split_backorder_po_by_brand) {
                        if (! isset($lines[$warehouseId][$product['brand_id']])) {
                            $lines[$warehouseId][$product['brand_id']] = [];
                        }
                        $lines[$warehouseId][$product['brand_id']][] = [
                            'product_id' => $product['product_id'],
                            'quantity' => $quantity_needed,
                        ];
                    } else {
                        if (! isset($lines[$warehouseId])) {
                            $lines[$warehouseId] = [];
                        }
                        $lines[$warehouseId][] = [
                            'product_id' => $product['product_id'],
                            'quantity' => $quantity_needed,
                        ];
                    }
                }
            }
        });

        return $lines;
    }

    public function applyDiscountLines(PurchaseOrder $purchaseOrder, float $discountRate): PurchaseOrder
    {
        $purchaseOrder->purchaseOrderLines->each(/**
         * @throws Throwable
         */ function (PurchaseOrderLine $purchaseOrderLine) use ($discountRate) {

            $purchaseOrderLine->discount_rate = $discountRate;
            $purchaseOrderLine->save();
        });

        return $purchaseOrder->refresh();
    }

    public function applyPricingTier(PurchaseOrder $purchaseOrder, SupplierPricingTier $supplierPricingTier): PurchaseOrder
    {
        $purchaseOrder->purchaseOrderLines->whereNotNull('product_id')->each(
            /**
             * @throws Throwable
             */
            function (PurchaseOrderLine $purchaseOrderLine) use ($supplierPricingTier) {

                $supplierProduct = $purchaseOrderLine->supplier_product;
                $supplierProductPrice = $supplierPricingTier
                    ->supplierProductPrices()
                    ->firstWhere('supplier_product_id', $supplierProduct->id);

                $purchaseOrderLine->amount = $supplierProductPrice->price;
                $purchaseOrderLine->save();
            }
        );

        return $purchaseOrder->refresh();
    }

    public function duplicateOrder($id)
    {
        $purchaseOrder = PurchaseOrder::with('purchaseOrderLines')->findOrFail($id);

        $clone = $purchaseOrder->replicate([
            'archived_at',
            'order_status',
            'submission_status',
            'receipt_status',
            'shipment_status',
            'invoice_status',
            'approved_at',
            'last_submitted_at',
            'last_supplier_confirmed_at',
            'fully_invoiced_at',
            'fully_shipped_at',
            'finalized_at',
            'sales_order_id',
        ]);
        $clone->sequence = PurchaseOrder::getNextSequence();
        $clone->purchase_order_number = PurchaseOrder::getLocalNumber($clone->sequence);
        $clone->order_status = PurchaseOrder::STATUS_DRAFT;
        $clone->purchase_order_date = now();
        $clone->other_date = now();
        $clone->cacheCurrencyRate(true);

        $clone->save();

        foreach ($purchaseOrder->purchaseOrderLines as $purchaseOrderLine) {
            $except = [
                'unreceived_quantity',
                'discount_amount_extended',
                'discount_amount',
                'total_amount',
            ];
            $newPurchaseOrderLine = $purchaseOrderLine->replicate($except);
            $attributes = $newPurchaseOrderLine->getAttributes();
            $newPurchaseOrderLine->setRawAttributes($attributes);
            $clone->purchaseOrderLines()->save($newPurchaseOrderLine);
        }

        return $clone;
    }

    /**
     * @param  PurchaseOrder  $purchaseOrder
     * @param  array  $payload
     * @return bool
     */
    public function changingProductLines(PurchaseOrder $purchaseOrder, array $payload): bool
    {

        if (!array_key_exists('purchase_order_lines', $payload)) {
            return false;
        }

        $currentProductLineIds = $purchaseOrder->purchaseOrderLines()->whereNotNull('product_id')
            ->pluck('id')
            ->toArray();
        $payloadLineIds = collect($payload['purchase_order_lines'])
            ->pluck('id')
            ->toArray();

        // All the current product line ids must exist in the new lines.
        if(!empty(array_diff($currentProductLineIds, $payloadLineIds))) {
            return true;
        }

        // All existing product lines exist in the new lines.
        $newLineIds = array_diff($payloadLineIds, $currentProductLineIds);
        if(empty($newLineIds)){
            return false;
        }

        // Ensure that none of the new lines is a product line.
        return collect($payload['purchase_order_lines'])
            ->whereIn('id', $newLineIds)
            ->contains(fn($line) => isset($line['product_id']));
    }
}
