<?php

namespace App\Services\PurchaseOrder\PurchaseOrderBuilder;

use App\Models\Supplier;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Builders\FillBackordersPurchaseOrderBuilder;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Builders\SalesBasedPurchaseOrderBuilder;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Builders\TargetStockLevelPurchaseOrderBuilder;
use Illuminate\Contracts\Container\BindingResolutionException;
use InvalidArgumentException;

class PurchaseOrderBuilderFactory
{
    const BUILDER_TYPE_TARGET_STOCK_LEVEL = 'target_stock_level';

    const BUILDER_TYPE_WITH_SALES_HISTORY = 'with_sales_history';

    const BUILDER_TYPE_FILL_BACKORDERS = 'fill_backorders';

    const SUPPORTED_BUILDER_TYPES = [
        self::BUILDER_TYPE_TARGET_STOCK_LEVEL,
        self::BUILDER_TYPE_WITH_SALES_HISTORY,
        self::BUILDER_TYPE_FILL_BACKORDERS,
    ];

    /**
     * @throws BindingResolutionException
     */
    public static function make(string $type, array $options): PurchaseOrderBuilder
    {
        self::checkInputs($type, $options);
        /** @var Supplier $supplier */
        $supplier = Supplier::with([])->findOrFail($options['supplier_id']);

        switch ($type) {
            case self::BUILDER_TYPE_TARGET_STOCK_LEVEL:
                return new TargetStockLevelPurchaseOrderBuilder(
                    $options['quantity'] ?? 0,
                    $supplier,
                    $options['destination_warehouse_id'],
                    $options['product_filters'] ?? [],
                    $options['use_moq'] ?? false
                );
            case self::BUILDER_TYPE_WITH_SALES_HISTORY:
                return new SalesBasedPurchaseOrderBuilder($supplier, $options);
            case self::BUILDER_TYPE_FILL_BACKORDERS:
                return new FillBackordersPurchaseOrderBuilder($supplier, $options);
            default:
                throw new InvalidArgumentException('Unsupported type: '.$type);
        }
    }

    private static function checkInputs(string $type, array $options)
    {
        if (! in_array($type, self::SUPPORTED_BUILDER_TYPES)) {
            throw new InvalidArgumentException('Unsupported type: '.$type);
        }

        if (! isset($options['supplier_id'])) {
            throw new InvalidArgumentException('supplier_id is required.');
        }
    }
}
