<?php

namespace App\Services\Product;

use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Repositories\BackorderQueueRepository;
use App\Services\InventoryManagement\BackorderManager;
use App\Services\InventoryManagement\PositiveInventoryEvent;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Collection;
use Throwable;

class ReleaseBackorderQueues extends ProductService
{
    /**
     * @var BackorderQueueRepository
     */
    protected $backorderQueues;

    private BackorderManager $backorderManager;

    /**
     * ReleaseBackorderQueues constructor.
     *
     *
     * @throws BindingResolutionException
     */
    public function __construct(Product $product)
    {
        parent::__construct($product);
        $this->backorderQueues = app()->make(BackorderQueueRepository::class);
        $this->backorderManager = app()->make(BackorderManager::class);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function execute(FifoLayer $fifoLayer, ?string $linkType = null, ?int $linkId = null, ?int $quantity = null)
    {
        if ($fifoLayer->link_type) {
            if ($fifoLayer->link && $fifoLayer->link instanceof PositiveInventoryEvent) {
                $this->backorderManager->releaseBackorderQueues(
                    new Collection([$fifoLayer->link])
                );
            } else {
                customlog('Broken-FifoLayer-Link', 'FifoLayer: '.$fifoLayer->id, [
                    'fifo' => $fifoLayer->toArray(),
                ]);
            }

            return;
        }

        $remainingQuantity = min($quantity ?? $fifoLayer->available_quantity, $fifoLayer->available_quantity);

        if ($remainingQuantity <= 0) {
            return;
        }

        // Get backorder queues and try to clear those that are satisfiable by the fifo layer
        $unreleasedBackorders = $this->backorderQueues->getActiveQueuesForProduct(
            $this->product->id,
            $fifoLayer->warehouse_id
        );

        /** @var BackorderQueue $backorderQueue */
        foreach ($unreleasedBackorders as $backorderQueue) {
            if ($backorderQueue->unreleased_quantity > $remainingQuantity) {
                $neededQuantity = $remainingQuantity;
            } else {
                $neededQuantity = $backorderQueue->unreleased_quantity;
            }

            // Release the needed quantity on the backorder queue.
            $release = $backorderQueue->addRelease(
                $linkId ?? $fifoLayer->link_id,
                $linkType ?? $fifoLayer->link_type,
                $neededQuantity,
                $fifoLayer
            );
            if ($release) {
                $remainingQuantity -= $release->released_quantity;
            }

            if ($remainingQuantity <= 0) {
                break;
            }
        }
    }
}
