<?php

namespace App\Services\InventoryManagement;

use App\Data\CreateBackorderQueueData;
use App\DTO\BackorderQueueCoverageDto;
use App\DTO\BackorderQueueDto;
use App\DTO\BackorderQueueReleaseDto;
use App\DTO\FifoLayerDto;
use App\DTO\InventoryMovementDto;
use App\DTO\ProductWarehousePairDto;
use App\DTO\SalesOrderLineLayerDto;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Models\BackorderQueueRelease;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Repositories\BackorderQueueCoverageRepository;
use App\Repositories\BackorderQueueRepository;
use App\Repositories\PurchaseOrderRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Spatie\LaravelData\Optional;
use Throwable;

class BackorderManager
{
    private BackorderQueueRepository $backorderQueueRepository;

    private BackorderQueueCoverageRepository $backorderQueueCoverageRepository;

    private PurchaseOrderRepository $purchaseOrderRepository;

    private bool $allowGlobalCoverage = false;

    public function __construct()
    {
        $this->backorderQueueRepository = app(BackorderQueueRepository::class);
        $this->backorderQueueCoverageRepository = app(BackorderQueueCoverageRepository::class);
        $this->purchaseOrderRepository = app(PurchaseOrderRepository::class);
    }

    public function withGlobalCoverage(): self
    {
        $this->allowGlobalCoverage = true;

        return $this;
    }

    public function coverBackorderQueues(?array $purchaseOrderLineIds = null, ?array $backorderQueueIds = null, ?array $productIds = null, ?int $warehouseId = null): void
    {
        if (! $this->allowGlobalCoverage) {
            if (empty($purchaseOrderLineIds) && empty($backorderQueueIds) && empty($productIds)) {
                return;
            }
        }

        if (empty($productIds) && ! empty($backorderQueueIds)) {
            $productIds = BackorderQueue::with('salesOrderLine')
                ->whereIn('id', $backorderQueueIds)
                ->get()
                ->pluck('salesOrderLine.product_id')
                ->unique()
                ->toArray();
        }

        $unreceivedPurchaseOrderLines = $this->purchaseOrderRepository->getUnreceivedPurchaseOrderLinesForOpenPurchaseOrders($purchaseOrderLineIds, $productIds, $warehouseId);
        $unreceivedPurchaseOrderLinesProductWarehousePairs = $this->purchaseOrderRepository->getUnreceivedPurchaseOrderLinesForOpenPurchaseOrdersProductWarehousePairs($purchaseOrderLineIds, $productIds, $warehouseId);

        // If no purchase order line ids or backorder queue ids were passed, we are syncing all coverages, so we can delete all unreleased coverages first
        // otherwise, we just add new coverages but keep the existing ones in place
        // TODO: This condition may not be passing often enough.  Need more thorough testing to see that coverages are getting properly updated for outlier cases.

        // Don't want to delete all if no products ids are passed because it is too much calculation work to do everything all over again until this is further optimized
        if (is_null($purchaseOrderLineIds) && is_null($backorderQueueIds) && ! is_null($productIds)) {
            $this->backorderQueueCoverageRepository->deleteUnreleased($productIds, $warehouseId);
        }

        $backorderQueueCoverageCollection = collect();

        // Add updates for any backorder queue coverages that are now released, release changed, or had release deleted
        $backorderQueueCoverageCollection = $backorderQueueCoverageCollection->merge($this->backorderQueueCoverageRepository->getCoveragesNeedingUpdate());
        customlog('backorderCoverages', 'Backorder coverages needing update', $backorderQueueCoverageCollection->toArray());

        $unreleasedUncoveredBackorders = $this->backorderQueueRepository->unreleasedUncoveredBackorders($unreceivedPurchaseOrderLinesProductWarehousePairs, $backorderQueueIds);

        // First, collect all unique product_id and warehouse_id pairs
        $productWarehousePairs = $this->backorderQueueRepository->unreleasedBackorderProductWarehouseGroups($unreceivedPurchaseOrderLinesProductWarehousePairs, $backorderQueueIds);

        $productWarehousePairs->each(function ($productWarehousePair) use (
            $unreleasedUncoveredBackorders,
            $backorderQueueCoverageCollection,
            $unreceivedPurchaseOrderLines,
        ) {
            $availablePurchaseOrderLines = $unreceivedPurchaseOrderLines
                ->where('product_id', $productWarehousePair->product_id)
                ->where('destination_warehouse_id', $productWarehousePair->warehouse_id)
                // Always return lines with estimated delivery dates first, then sort by purchase order date
                ->sortBy(function ($line) {
                    if (! is_null($line->estimated_delivery_date)) {
                        return [$line->estimated_delivery_date, $line->purchase_order_date];
                    }

                    return [Carbon::now()->addYears(100), $line->purchase_order_date];
                });
            if ($availablePurchaseOrderLines->count() == 0) {
                return;
            }

            $backordersNeedingCoverage = $unreleasedUncoveredBackorders
                ->where('product_id', $productWarehousePair->product_id)
                ->where('warehouse_id', $productWarehousePair->warehouse_id);
            //->sortBy('priority')
            //->sortBy('backorder_date');

            $backordersNeedingCoverage
                ->each(function ($backorderQueue) use (
                    $availablePurchaseOrderLines,
                    $backorderQueueCoverageCollection
                ) {
                    $backorderQueueQtyNeedingCoverage = $backorderQueue->backordered_quantity - $backorderQueue->released_quantity - $backorderQueue->covered_quantity;

                    while ($backorderQueueQtyNeedingCoverage > 0 && $availablePurchaseOrderLines->count() > 0) {
                        /** @var PurchaseOrderLine $purchaseOrderLine */
                        $purchaseOrderLine = $availablePurchaseOrderLines->shift();

                        // To get the coverage quantity for this PO line, we get the difference
                        // between the unreceived quantity and existing coverages. This gives
                        // the residual quantity (or reduction in coverage if negative) for the line.
                        $existingCoverages = $purchaseOrderLine->coveredBackorderQueues()
                            ->sum('unreleased_quantity');
                        $residual = (int) $purchaseOrderLine->unreceived_quantity - $existingCoverages;

                        if ($residual != 0) {
                            $currentCoverageQty = $purchaseOrderLine
                                ->coveredBackorderQueues()
                                ->where('backorder_queue_id', $backorderQueue->id)
                                ->sum('covered_quantity');

                            // The PO can still cover more
                            $additionalCoverageQty = min($backorderQueueQtyNeedingCoverage, $residual);
                            $backorderCoverage = $this->addBackorderQueueCoverageDto(
                                $backorderQueue->id,
                                $purchaseOrderLine->id,
                                $additionalCoverageQty + $currentCoverageQty
                            );

                            // Update quantities
                            $backorderQueueQtyNeedingCoverage -= $additionalCoverageQty;
                            $purchaseOrderLine->unreceived_quantity -= $additionalCoverageQty;

                            $backorderQueueCoverageCollection->add($backorderCoverage);

                            // If the purchase order line still has remaining unreceived quantity, add it back to the top of available purchase order lines
                            if (max(0, $purchaseOrderLine->unreceived_quantity - $purchaseOrderLine->backorder_queue_coverage_quantity) > 0) {
                                $availablePurchaseOrderLines->prepend($purchaseOrderLine);
                            }
                        }
                    }
                });

            if ($backorderQueueCoverageCollection->count() >= 1000) {
                $this->backorderQueueCoverageRepository->saveBulk($backorderQueueCoverageCollection);
                // Reset the collection
                $backorderQueueCoverageCollection->splice(0);
            }
        });

        // After the loop - save any remaining items in the collection
        if (! $backorderQueueCoverageCollection->isEmpty()) {
            $this->backorderQueueCoverageRepository->saveBulk($backorderQueueCoverageCollection);
        }
    }

    private function addBackorderQueueCoverageDto(
        int $backorderQueueId,
        int $purchaseOrderLineId,
        int $coverageQuantity
    ): BackorderQueueCoverageDto {
        return BackorderQueueCoverageDto::from([
            'backorder_queue_id' => $backorderQueueId,
            'purchase_order_line_id' => $purchaseOrderLineId,
            'covered_quantity' => $coverageQuantity,
            'released_quantity' => 0,
            'created_at' => now(),
        ]);
    }

    /**
     * @throws Throwable
     */
    public function releaseBackorderQueues(Collection|EloquentCollection $positiveInventoryEventCollection): Collection
    {
        if ($positiveInventoryEventCollection->isEmpty()) {
            return collect();
        }

        // Handle priority releases.
        // This is for purchase order receipts with tight coverages
        // to specific backorder queues.
        $positiveInventoryEventCollection = $this->releaseTightCoverages($positiveInventoryEventCollection);

        $fifoLayers = $positiveInventoryEventCollection
            ->map(fn (PositiveInventoryEvent $positiveInventoryEvent) => $positiveInventoryEvent->getFifoLayer());

        $productWarehousePairs = $fifoLayers->map(function (FifoLayer $fifoLayer) {
            return ProductWarehousePairDto::from([
                'product_id' => $fifoLayer->product_id,
                'warehouse_id' => $fifoLayer->warehouse_id,
            ]);
        })->unique();

        // Get all backorder queues that have not been released, ordered by priority
        $backorders = $this->backorderQueueRepository->getUnreleasedBackordersForProductWarehousePairs($productWarehousePairs);

        $movementInserts = [];
        $movementDeletes = [];
        $fifoUpdates = [];
        $backorderUpdates = [];
        $backorderReleaseInserts = [];
        $backorderCoveragesUpdates = [];
        $backorderCoverageDeletes = [];
        $salesOrderLineLayersInserts = [];

        /** @var BackorderQueue $backorder */
        foreach ($backorders as $backorder) {
            // Each sales order line should have one active backorder queue movement,
            // we use that as a model to recreate the movements.
            /** @var InventoryMovement $backorderActiveMovement */
            $backorderActiveMovement = $backorder->salesOrderLine
                ->inventoryMovements
                ->where('layer_type', BackorderQueue::class)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->first();
            if (! $backorderActiveMovement) {
                continue;
            } // TODO:: Handle this situation (write to log, maybe).

            // Build up the needed data for inserts and updates
            // Get fifo layers that can fulfill this backorder queue
            $fifoAllocations = $this->getFifoLayerAllocationsFor(
                quantity: $backorder->shortage_quantity,
                backorder: $backorder,
                fifoLayers: $fifoLayers,
                fifoUpdates: $fifoUpdates
            );
            // Add the allocations to the fifo updates
            array_push($fifoUpdates, ...$fifoAllocations);

            // Next, we create the inventory movement data based on the fifo layer allocations
            $baseline = $backorderActiveMovement->replicate(['id'])->toArray();
            [$inserts, $deletes] = $this->makeInventoryMovementData($backorder, $baseline, $fifoAllocations);
            array_push($movementInserts, ...$inserts);
            array_push($movementDeletes, ...$deletes);

            // Next, we build the backorder update data
            $backorderUpdates[] = [
                'id' => $backorder->id,
                'released_quantity' => $backorder->released_quantity + collect($fifoAllocations)->sum('fulfilled_quantity'),
                'updated_at' => Carbon::now(),
            ];

            // Next, we build the backorder release insert data.
            foreach ($fifoAllocations as $allocation) {
                $backorderReleaseInserts[] = [
                    'backorder_queue_id' => $backorder->id,
                    'link_type' => $allocation['link_type'],
                    'link_id' => $allocation['link_id'],
                    'reference' => $allocation['reference'],
                    'released_quantity' => $allocation['fulfilled_quantity'],
                    'created_at' => Carbon::now(),
                ];

                $salesOrderLineLayersInserts[] = [
                    'sales_order_line_id' => $backorder->sales_order_line_id,
                    'quantity' => $allocation['fulfilled_quantity'],
                    'layer_id' => $allocation['id'],
                    'layer_type' => FifoLayer::class,
                    'created_at' => Carbon::now(),
                ];
            }

            // We build the backorder coverages update data.
            // For coverages that are part of the release, we
            // update their released quantities. However, for
            // coverages that aren't part of the release, we
            // ensure that their covered quantities are in sync.
            $purchaseReceipts = $positiveInventoryEventCollection
                ->filter(fn ($event) => $event instanceof PurchaseOrderShipmentReceiptLine);
            [$updates, $deletes] = $this->syncBackorderCoverageQuantities(
                $backorder,
                $purchaseReceipts,
                end($backorderUpdates),
                $fifoAllocations
            );

            array_push($backorderCoveragesUpdates, ...$updates);
            array_push($backorderCoverageDeletes, ...$deletes);
        }

        // Finally, we ensure that we account for already used quantities in the fifo layer updates.
        $affectedLayers = $fifoLayers->whereIn('id', collect($fifoUpdates)->pluck('id')->toArray());
        $groupedFifoLayerUpdates = [];
        /** @var FifoLayer $fifoLayer */
        foreach ($affectedLayers as $fifoLayer) {
            $released = collect($fifoUpdates)->where('id', $fifoLayer->id)->sum('fulfilled_quantity');
            $groupedFifoLayerUpdates[] = [
                'id' => $fifoLayer->id,
                'fulfilled_quantity' => min($released + $fifoLayer->fulfilled_quantity, $fifoLayer->original_quantity),
                'updated_at' => Carbon::now(),
            ];
        }

        // Updates and inserts
        DB::transaction(function () use (
            $backorderUpdates,
            $groupedFifoLayerUpdates,
            $backorderReleaseInserts,
            $movementInserts,
            $movementDeletes,
            $salesOrderLineLayersInserts,
            $backorderCoveragesUpdates,
            $backorderCoverageDeletes
        ) {
            batch()->update(new BackorderQueue, $backorderUpdates, 'id');
            batch()->update(new FifoLayer, $groupedFifoLayerUpdates, 'id');
            batch()->update(new BackorderQueueCoverage, $backorderCoveragesUpdates, 'id');
            BackorderQueueRelease::query()->insert($backorderReleaseInserts);
            BackorderQueueCoverage::query()->whereIn('id', array_column($backorderCoverageDeletes, 'id'))->delete();

            // Inventory movements
            InventoryMovement::query()->whereIn('id', $movementDeletes)->delete();
            InventoryMovement::query()->insert($movementInserts);

            // Sales order line layers
            SalesOrderLineLayer::query()->insert($salesOrderLineLayersInserts);
        });

        // Sync backorder queue coverages.
        $backorderIds = $backorders->pluck('id')->toArray();
        if (! empty($backorderIds)) {
            dispatch(
                new SyncBackorderQueueCoveragesJob(
                    backorderQueueIds: $backorderIds
                )
            );
        }

        return $backorders;
    }

    /**
     * Creates a backorder queue and manages the allocation and coverage process.
     *
     * @param int $quantity The quantity of items for backordering.
     * @param SalesOrderLine $salesOrderLine The sales order line associated with the backorder.
     * @param InventoryReductionCause $reductionCause The cause of inventory reduction, defaults to increased negative event.
     * @param int|null $supplierId Optional supplier ID.
     * @param InventoryEvent|null $event Optional inventory event.
     * @return BackorderQueue The created backorder queue with coverages.
     */
    public function createBackorderQueue(
        int $quantity,
        SalesOrderLine $salesOrderLine,
        InventoryReductionCause $reductionCause = InventoryReductionCause::INCREASED_NEGATIVE_EVENT,
        ?int $supplierId = null,
        InventoryEvent $event = null
    ): BackorderQueue {

        // Initialize the backorder queue with the given parameters.
        $queue = $this->backorderQueueRepository->withSupplier(
            $quantity,
            $salesOrderLine,
            $reductionCause,
            $supplierId
        );

        // Handle the special case of a reduced positive inventory event.
        if ($reductionCause == InventoryReductionCause::REDUCED_POSITIVE_EVENT) {
            return $this->syncCoveragesAndReleasesAfterPositiveEventReduction($queue, $quantity, $event);
        }

        // Retrieve available purchase order lines that can be used for coverage.
        $purchaseOrderLines = $this->backorderQueueRepository->getAvailablePurchaseOrderLinesForCoverages($queue);

        // Initialize containers for coverage data to be inserted or updated.
        $coverageInserts = [];
        $coverageUpdates = [];

        // Fetch existing coverages related to the queue.
        $existingCoverages = $this->backorderQueueCoverageRepository->getExistingCoverages(
            $queue,
            $purchaseOrderLines->pluck('id')->toArray()
        );

        // Determine the total quantity required to be covered.
        $totalNeeded = $queue->unallocated_backorder_quantity;

        // Process each purchase order line for potential coverage.
        foreach ($purchaseOrderLines as $purchaseOrderLine) {
            // Determine the coverage quantity based on availability and need.
            if ($purchaseOrderLine->quantity_available >= $totalNeeded) {
                $quantity = $totalNeeded;
                $totalNeeded = 0;
            } else {
                $quantity = $purchaseOrderLine->quantity_available;
                $totalNeeded -= $quantity;
            }

            // Check if there's existing coverage to update, otherwise create new.
            $coverage = $existingCoverages->where('purchase_order_line_id', $purchaseOrderLine->id)->first();
            if ($coverage) {
                // Update existing coverage.
                $coverageUpdates[] = [
                    'id' => $coverage->id,
                    'covered_quantity' => $coverage->covered_quantity + $quantity,
                    'updated_at' => Carbon::now(),
                ];
            } else {
                // Create new coverage entry.
                $coverageInserts[] = [
                    'purchase_order_line_id' => $purchaseOrderLine->id,
                    'covered_quantity' => $quantity,
                    'released_quantity' => 0,
                    'backorder_queue_id' => $queue->id,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ];
            }

            // Break the loop if no more coverage is needed.
            if ($totalNeeded <= 0) {
                break;
            }
        }

        // Perform batch updates and inserts for coverages.
        if (!empty($coverageUpdates)) {
            batch()->update(new BackorderQueueCoverage, $coverageUpdates, 'id');
        }
        if (!empty($coverageInserts)) {
            BackorderQueueCoverage::query()->insert($coverageInserts);
        }

        // Return the updated queue with loaded coverages.
        return $queue->loadMissing('backorderQueueCoverages');
    }



    /**
     * Synchronizes coverages and releases in the backorder queue after a reduction
     * in positive inventory events.
     *
     * @param BackorderQueue $queue The backorder queue to be updated.
     * @param int $quantity The quantity by which the inventory event is reduced.
     * @param PositiveInventoryEvent|null $originatingEvent The originating positive inventory event.
     * @return BackorderQueue Updated backorder queue.
     */
    public function syncCoveragesAndReleasesAfterPositiveEventReduction(
        BackorderQueue $queue,
        int $quantity,
        PositiveInventoryEvent|null $originatingEvent
    ): BackorderQueue {
        // Check if the originating event is null. If so, no action is required.
        if (!$originatingEvent) {
            return $queue;
        }

        // Handle reduction for a Purchase Order Shipment Receipt Line event.
        if ($originatingEvent instanceof PurchaseOrderShipmentReceiptLine) {
            // Retrieve coverages associated with the backorder queue that are affected by the event.
            $coverages = $queue->backorderQueueCoverages()
                ->whereHas('purchaseOrderLine.purchaseOrderShipmentReceiptLines', function($query) use ($originatingEvent) {
                    return $query->where('purchase_order_shipment_receipt_lines.id', $originatingEvent->id);
                })
                ->with(['backorderQueue.salesOrderLine'])
                ->get()
                ->where('backorderQueue.salesOrderLine.unfulfilled_quantity', '>', 0);

            // Initialize variables for updating coverage release quantities.
            $coverageTotal = $quantity;
            $coverageReleaseUpdates = [];

            // Iterate over each coverage to adjust the released quantity.
            foreach ($coverages as $coverage) {
                $unfulfilledQty = $coverage->backorderQueue->salesOrderLine->unfulfilled_quantity;
                $applicable = min($coverageTotal, $unfulfilledQty);

                // Calculate the new released quantity and track the updates.
                if ($coverage->released_quantity >= $applicable) {
                    $coverageReleaseUpdates[] = [
                        'id' => $coverage->id,
                        'released_quantity' => $coverage->released_quantity - $applicable,
                        'updated_at' => Carbon::now(),
                    ];

                    $coverageTotal -= $applicable;
                } else {
                    $coverageReleaseUpdates[] = [
                        'id' => $coverage->id,
                        'released_quantity' => 0,
                        'updated_at' => Carbon::now(),
                    ];
                    $coverageTotal -= $coverage->released_quantity;
                }

                // Break the loop if no more quantity needs to be adjusted.
                if ($coverageTotal <= 0) {
                    break;
                }
            }

            // Apply the updates to the coverages.
            batch()->update(new BackorderQueueCoverage, $coverageReleaseUpdates, 'id');
        }

        // Handle the release updates after adjusting the coverages.
        $releaseUpdates = [];
        $releaseDeletes = [];

        $releases = $originatingEvent->backorderQueueReleases()->get();

        // Iterate over each release to adjust or delete based on the reduced quantity.
        foreach ($releases as $release) {
            if ($release->released_quantity > $quantity) {
                $releaseUpdates[] = [
                    'id' => $release->id,
                    'released_quantity' => $release->released_quantity - $quantity,
                    'updated_at' => Carbon::now(),
                ];
                break;
            } else {
                $releaseDeletes[] = $release->id;
                $quantity -= $release->released_quantity;
            }

            // Break the loop if no more quantity needs to be adjusted.
            if ($quantity <= 0) {
                break;
            }
        }

        // Apply updates and deletions to the releases.
        batch()->update(new BackorderQueueRelease, $releaseUpdates, 'id');
        BackorderQueueRelease::query()->whereIn('id', $releaseDeletes)->delete();

        // Refresh and return the updated backorder queue.
        return $queue->refresh();
    }



    private function syncBackorderCoverageQuantities(
        BackorderQueue $backorder,
        Collection $purchaseReceipts,
        array $backorderUpdate,
        array $fifoAllocations
    ): array {
        $updates = [];
        $deletes = [];

        // Note that we're using the released quantity in the update
        // as that's more recent from the release.
        $shortageQuantity = max($backorder->backordered_quantity - $backorderUpdate['released_quantity'], 0);

        $coverages = $backorder->backorderQueueCoverages;

        $coverageReleases = [];

        if ($purchaseReceipts->isNotEmpty()) {
            /** @var PurchaseOrderShipmentReceiptLine $receiptLine */
            foreach ($purchaseReceipts as $receiptLine) {
                $orderLIneId = $receiptLine->purchaseOrderShipmentLine->purchase_order_line_id;

                $receiptCoverage = $coverages
                    ->where('purchase_order_line_id', $orderLIneId)
                    ->first();

                if ($receiptCoverage) {
                    $coverageReleases[] = [
                        'id' => $receiptCoverage->id,
                        'released_quantity' => collect($fifoAllocations)
                            ->where('link_id', $receiptLine->id)
                            ->where('link_type', PurchaseOrderShipmentReceiptLine::class)
                            ->sum('fulfilled_quantity'),
                        'purchase_order_line_id' => $orderLIneId,
                    ];
                }
            }
        }

        // We sync the coverages and their quantities.
        /** @var BackorderQueueCoverage $coverage */
        foreach ($coverages as $coverage) {
            // We need to account for releases for the coverage in memory, not
            // yet reflected in the coverage object.
            $freshCoverageReceipt = collect($coverageReleases)
                ->where('purchase_order_line_id', $coverage->purchase_order_line_id)
                ->sum('released_quantity');

            $totalUnreleased = max($coverage->unreleased_quantity - $freshCoverageReceipt, 0);

            if ($totalUnreleased > $shortageQuantity) {
                // This is now an over-coverage. we need to update the coverage quantity.
                $overage = min($coverage->unreleased_quantity - $shortageQuantity, $coverage->covered_quantity);
                $updates[] = [
                    'id' => $coverage->id,
                    'covered_quantity' => $coverage->covered_quantity - $overage,
                    'released_quantity' => $coverage->released_quantity,
                ];
                $shortageQuantity -= end($updates)['covered_quantity'];
            }
            if ($shortageQuantity <= 0) {
                // Entire shortage quantity is covered, we mark any extra coverages for deletion.
                $ids = collect($updates)->pluck('id')->merge(
                    collect($coverageReleases)->pluck('id')
                )->toArray();
                $deletes = $coverages->whereNotIn('id', $ids)->toArray();
                break;
            }
        }

        $updates = collect($updates)->map(function ($update) use ($coverageReleases) {
            $released = collect($coverageReleases)->where('id', $update['id']);
            if ($released->isNotEmpty()) {
                $update['released_quantity'] = $update['released_quantity'] + $released->sum('released_quantity');
            }

            return $update;
        });

        return [$updates, $deletes];
    }

    private function makeInventoryMovementData(BackorderQueue $backorder, array $baseline, array $fifoAllocations): array
    {
        $inserts = [];

        // For each fifo allocation, we build a corresponding set of inventory movements.
        foreach ($fifoAllocations as $allocation) {
            $inserts[] = array_merge($baseline, [
                'quantity' => -$allocation['fulfilled_quantity'],
                'layer_id' => $allocation['id'],
                'layer_type' => FifoLayer::class,
                'inventory_movement_date' => Carbon::parse($baseline['inventory_movement_date'])->format('Y-m-d H:i:s'),
                'created_at' => Carbon::now(),
            ]);

            // Add reservation
            $inserts[] = array_merge($baseline, [
                'quantity' => $allocation['fulfilled_quantity'],
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
                'layer_id' => $allocation['id'],
                'layer_type' => FifoLayer::class,
                'inventory_movement_date' => Carbon::parse($baseline['inventory_movement_date'])->format('Y-m-d H:i:s'),
                'created_at' => Carbon::now(),
            ]);
        }

        // If we still have a shortage, we add in an extra movement pair for the backorder queue.
        $residual = max(0, $backorder->shortage_quantity - collect($fifoAllocations)->sum('fulfilled_quantity'));

        if ($residual > 0) {
            $inserts[] = array_merge($baseline, [
                'quantity' => -$residual,
                'inventory_movement_date' => Carbon::parse($baseline['inventory_movement_date'])->format('Y-m-d H:i:s'),
                'created_at' => Carbon::now(),
            ]);

            // Add reservation
            $inserts[] = array_merge($baseline, [
                'quantity' => $residual,
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
                'inventory_movement_date' => Carbon::parse($baseline['inventory_movement_date'])->format('Y-m-d H:i:s'),
                'created_at' => Carbon::now(),
            ]);
        }

        // Mark existing backorder movement ids for deletion.
        $deletes = $backorder->salesOrderLine
            ->inventoryMovements
            ->where('layer_type', BackorderQueue::class)
            ->pluck('id')
            ->toArray();

        return [$inserts, $deletes];
    }

    private function getFifoLayerAllocationsFor(
        int $quantity,
        BackorderQueue $backorder,
        Collection $fifoLayers,
        array $fifoUpdates,
    ): array {
        if ($quantity <= 0) {
            return [];
        }

        $unallocated = $fifoLayers
            ->where('product_id', $backorder->salesOrderLine->product_id)
            ->where('warehouse_id', $backorder->salesOrderLine->warehouse_id)
            ->sortBy('fifo_layer_date')
            ->filter(fn ($fifoLayer) => $fifoLayer->available_quantity > 0);

        $allocations = [];

        /** @var FifoLayer $fifoLayer */
        foreach ($unallocated as $fifoLayer) {
            $alreadyUsedQuantity = collect($fifoUpdates)->where('id', $fifoLayer->id)->sum('fulfilled_quantity');
            $availableFifoQuantity = $fifoLayer->available_quantity - $alreadyUsedQuantity;

            if ($availableFifoQuantity > 0) {
                $quantityToRelease = min($quantity, $availableFifoQuantity);

                // It's important to know that we're only returning the quantity used
                // to release the backorder, so updates to the fifo layer should consider
                // already fulfilled_quantity as well.
                $allocations[] = [
                    'id' => $fifoLayer->id,
                    'fulfilled_quantity' => $quantityToRelease,
                    'original_quantity' => $fifoLayer->original_quantity,
                    'link_type' => $fifoLayer->link_type,
                    'link_id' => $fifoLayer->link_id,
                    'reference' => $fifoLayer->link_reference,
                    'updated_at' => Carbon::now(),
                ];

                $quantity -= $availableFifoQuantity;
            }

            if ($quantity <= 0) {
                break;
            }
        }

        return $allocations;
    }

    private function addBackorderQueueDto(BackorderQueue $backorderQueue, int $releaseQuantity): BackorderQueueDto
    {
        return BackorderQueueDto::from([
            'id' => $backorderQueue->id,
            'released_quantity' => $backorderQueue->released_quantity + $releaseQuantity,
        ]);
    }

    private function addFifoLayerDto(int $fifoLayerId, int $releaseQuantity): FifoLayerDto
    {
        return FifoLayerDto::from([
            'id' => $fifoLayerId,
            'fulfilled_quantity' => $releaseQuantity,
        ]);
    }

    private function addInventoryMovementDto(InventoryMovement $inventoryMovement, Collection $fifoLayerAllocations, Collection $inventoryMovementInserts): InventoryMovementDto
    {
        $qtySign = ($inventoryMovement->inventory_status == InventoryMovement::INVENTORY_STATUS_ACTIVE ? -1 : 1);
        /*
         * We skip the first element because the first element is to update the existing inventory movement record
         */
        $fifoLayerAllocations->skip(1)->each(function ($fifoLayerAllocation) use ($inventoryMovementInserts, $qtySign, $inventoryMovement) {
            $inventoryMovementInserts->add(InventoryMovementDto::from([
                'layer_id' => $fifoLayerAllocation['layer_id'],
                'layer_type' => FifoLayer::class,
                'product_id' => $inventoryMovement->product_id,
                'warehouse_id' => $inventoryMovement->warehouse_id,
                'quantity' => $qtySign * $fifoLayerAllocation['quantity'],
                'inventory_status' => $inventoryMovement->inventory_status,
                'link_type' => $inventoryMovement->link_type,
                'link_id' => $inventoryMovement->link_id,
                'reference' => $inventoryMovement->reference,
                'type' => $inventoryMovement->type,
                'inventory_movement_date' => $inventoryMovement->inventory_movement_date,
                'created_at' => Carbon::now(),
            ]));
        });

        return InventoryMovementDto::from([
            'id' => $inventoryMovement->id,
            'layer_id' => $fifoLayerAllocations->first()['layer_id'],
            'layer_type' => FifoLayer::class,
            'quantity' => $qtySign * $fifoLayerAllocations->first()['quantity'],
            'updated_at' => Carbon::now(),
        ]);
    }

    private function addBackorderQueueReleaseDto(BackorderQueue $backorderQueue, FifoLayer $fifoLayerAvailable, int $releaseQuantity): BackorderQueueReleaseDto
    {
        return BackorderQueueReleaseDto::from([
            'backorder_queue_id' => $backorderQueue->id,
            'link_id' => $fifoLayerAvailable->link_id,
            'link_type' => $fifoLayerAvailable->link_type,
            'released_quantity' => $releaseQuantity,
            'reference' => $fifoLayerAvailable->link->getReference(), // getReference is a method in the InventoryEvent interface
            'created_at' => Carbon::now(),
        ]);
    }

    private function addSalesOrderLineLayerDto(BackorderQueue $backorderQueue, array $fifoLayerAllocation): SalesOrderLineLayerDto
    {
        return SalesOrderLineLayerDto::from([
            'sales_order_line_id' => $backorderQueue->sales_order_line_id,
            'layer_id' => $fifoLayerAllocation['layer_id'],
            'layer_type' => FifoLayer::class,
            'quantity' => $fifoLayerAllocation['quantity'],
            'created_at' => Carbon::now(),
        ]);
    }

    /**
     * @throws Throwable
     */
    private function releaseTightCoverages(EloquentCollection $positiveEvents): EloquentCollection|Collection
    {
        $purchaseReceipts = $positiveEvents->filter(fn ($event) => $event instanceof PurchaseOrderShipmentReceiptLine);
        if ($purchaseReceipts->isEmpty()) {
            return $positiveEvents;
        }

        $remainingEvents = $positiveEvents->filter(fn ($event) => ! ($event instanceof PurchaseOrderShipmentReceiptLine));

        $movementInserts = [];
        $movementDeletes = [];
        $fifoUpdates = [];
        $backorderUpdates = [];
        $backorderReleaseInserts = [];
        $backorderCoveragesUpdates = [];
        $salesOrderLineLayersInserts = [];

        $purchaseReceipts->load(['purchaseOrderLine.coveredBackorderQueues' => function ($q) {
            return $q->where('is_tight', true)->where('unreleased_quantity', '>', 0);
        }]);

        /** @var PurchaseOrderShipmentReceiptLine|PositiveInventoryEvent $receiptLine */
        foreach ($purchaseReceipts as $receiptLine) {
            $coverages = $receiptLine->purchaseOrderLine->coveredBackorderQueues;
            $fifoLayer = $receiptLine->getFifoLayer();

            /** @var BackorderQueueCoverage $coverage */
            foreach ($coverages as $coverage) {
                $quantity = min($coverage->unreleased_quantity, $receiptLine->quantity, $fifoLayer->available_quantity);

                // Records
                $backorderActiveMovement = $coverage->backorderQueue->salesOrderLine
                    ->inventoryMovements
                    ->where('layer_type', BackorderQueue::class)
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                    ->first();
                if (! $backorderActiveMovement) {
                    continue;
                }

                // Add the fifo charge to the fifo updates
                $fifoUpdates[] = [
                    'id' => $fifoLayer->id,
                    'fulfilled_quantity' => min($fifoLayer->fulfilled_quantity + $quantity, $fifoLayer->original_quantity),
                ];

                // Next, we create the inventory movement data based on the fifo layer allocations
                $baseline = $backorderActiveMovement->replicate(['id'])->toArray();
                [$inserts, $deletes] = $this->makeInventoryMovementData($coverage->backorderQueue, $baseline, [end($fifoUpdates)]);
                array_push($movementInserts, ...$inserts);
                array_push($movementDeletes, ...$deletes);

                // Next, we build the backorder update data
                $backorderUpdates[] = [
                    'id' => $coverage->backorderQueue->id,
                    'released_quantity' => $coverage->backorderQueue->released_quantity + $quantity,
                ];

                // Next, we build the backorder release insert data.
                $backorderReleaseInserts[] = [
                    'backorder_queue_id' => $coverage->backorderQueue->id,
                    'link_type' => $receiptLine::class,
                    'link_id' => $receiptLine->id,
                    'reference' => $receiptLine->getReference(),
                    'released_quantity' => $quantity,
                    'created_at' => Carbon::now(),
                ];

                $salesOrderLineLayersInserts[] = [
                    'sales_order_line_id' => $coverage->backorderQueue->sales_order_line_id,
                    'quantity' => $quantity,
                    'layer_id' => $fifoLayer->id,
                    'layer_type' => FifoLayer::class,
                    'created_at' => Carbon::now(),
                ];

                $backorderCoveragesUpdates[] = [
                    'id' => $coverage->id,
                    'released_quantity' => $coverage->released_quantity + $quantity,
                    'updated_at' => Carbon::now(),
                ];

                $receiptLine->quantity -= $quantity;

                if ($receiptLine->quantity <= 0) {
                    break;
                }
            }

            if ($receiptLine->quantity > 0) {
                // Receipt still has quantity after tight coverage release,
                // we put it back into the remaining events to potentially be
                // used to release other backorder queues.
                $remainingEvents->add($receiptLine);
            }
        }

        // Updates and inserts
        DB::transaction(function () use (
            $backorderUpdates,
            $backorderReleaseInserts,
            $movementInserts,
            $movementDeletes,
            $salesOrderLineLayersInserts,
            $backorderCoveragesUpdates,
            $fifoUpdates
        ) {
            batch()->update(new BackorderQueue, $backorderUpdates, 'id');
            batch()->update(new FifoLayer, $fifoUpdates, 'id');
            batch()->update(new BackorderQueueCoverage, $backorderCoveragesUpdates, 'id');
            BackorderQueueRelease::query()->insert($backorderReleaseInserts);

            // Inventory movements
            InventoryMovement::query()->whereIn('id', $movementDeletes)->delete();
            InventoryMovement::query()->insert($movementInserts);

            // Sales order line layers
            SalesOrderLineLayer::query()->insert($salesOrderLineLayersInserts);
        });

        return $remainingEvents;
    }

    // Get unreleased coverages and releases for a given backorder queue
    public function getHistory(BackorderQueue $backorderQueue): array
    {
        return [
            'unreleased_coverages' => $this->backorderQueueRepository->getUnreleasedCoveragesForBackorderQueue($backorderQueue),
            'releases' => $backorderQueue->backorderQueueReleases,
        ];
    }

    public function getHistoryAndNextScheduledPurchaseOrder(BackorderQueue $backorderQueue): array
    {
        return array_merge($this->getHistory($backorderQueue), [
            'next_scheduled_po' => $backorderQueue->getSchedule()
        ]);
    }

    public function newCreateBackorder(CreateBackorderQueueData $data): BackorderQueue
    {
        if ($data->backorder_date instanceof Optional) {
            $data->backorder_date = now();
        }
        if ($data->priority instanceof Optional) {
            $data->priority = $this->backorderQueueRepository->getNextPriority($data->salesOrderLine);
        }

        $salesOrderLine = $data->salesOrderLine;

        $data = $data->except('salesOrderLine');

        $backorderQueue = new BackorderQueue($data->toArray());
        $backorderQueue->sales_order_line_id = $salesOrderLine->id;
        $backorderQueue->save();

        // TODO: Coverages logic here
        return $backorderQueue;
    }
}
