<?php

namespace App\Services\InventoryForecasting;

use App\DTO\PurchaseOrderDto;
use App\DTO\PurchaseOrderLineDto;
use App\Helpers;
use App\Models\ForecastItemPOLineLink;
use App\Models\PurchaseOrder;
use App\Models\Setting;
use App\Repositories\CurrencyRepository;
use App\Repositories\PurchaseOrderRepository;
use App\Repositories\SupplierRepository;
use Illuminate\Database\Eloquent\Collection;
use Spatie\LaravelData\DataCollection;

class PurchaseOrderBuilder
{

    /**
     * @param  PurchaseOrderRepository  $orders
     * @param  SupplierRepository  $suppliers
     * @param  CurrencyRepository  $currencies
     */
    public function __construct(
        private readonly PurchaseOrderRepository $orders,
        private readonly SupplierRepository $suppliers,
        private readonly CurrencyRepository $currencies
    ){}

    /**
     * @param  array  $payload
     * @return Collection|\Illuminate\Support\Collection|PurchaseOrder
     * @throws \Throwable
     */
    public function create(array $payload): Collection|\Illuminate\Support\Collection|PurchaseOrder
    {
        $results = $this->orders->saveWithRelations($this->preparePayload($payload));
        if ($results instanceof PurchaseOrder){
            $results = Collection::make([$results]);
        }

        // Link the purchase order lines with their respective
        // forecast items.
        $links = [];
        foreach($results as $index => $order){
            $forecastItemIds = array_column($payload[$index]['purchase_order_lines'], 'forecast_item_id');
            $purchaseOrderLineIds = array_column($order->purchaseOrderLines->toArray(), 'id');
            array_push($links, ...array_map(function($forecastItemId, $purchaseOrderLineId){
                return [
                    'forecast_item_id' => $forecastItemId,
                    'purchase_order_line_id' => $purchaseOrderLineId
                ];
            }, $forecastItemIds, $purchaseOrderLineIds));
        }

        ForecastItemPOLineLink::query()->insert($links);

        return $results;
    }

    /**
     * @param  array  $payload
     * @return DataCollection
     */
    private function preparePayload(array $payload): DataCollection
    {

        // Get default destination warehouse for purchase orders
        $warehouseId = Setting::getValueByKey(
            Setting::KEY_PO_DEFAULT_WAREHOUSE
        );

        // Get the prefix and number of digits
        $prefix = Helpers::setting(Setting::KEY_PO_PREFIX, '');
        $numOfDigits = Helpers::setting(Setting::KEY_PO_NUM_DIGITS, 5);
        $nextSequence = PurchaseOrder::getNextSequence();

        $now = now();

        $supplierDefaultCurrencies = $this->suppliers->getDefaultCurrenciesForSuppliers(
            supplierIds: array_column($payload, 'supplier_id')
        );
        $supplierDefaultCurrencies = array_combine(
            array_column($supplierDefaultCurrencies, 'id'),
            $supplierDefaultCurrencies
        );
        $defaultCurrency = $this->currencies->getDefault();

        // Get purchase order nominal code
        $purchaseOrderNominalCode = Setting::getValueByKey(
            Setting::KEY_NC_MAPPING_COGS
        );

        $defaultStore = Setting::getValueByKey(
            Setting::KEY_PO_DEFAULT_STORE
        );

        $data = collect();

        foreach ($payload as $key => $order) {
            $payload[$key]['sequence'] = $nextSequence;
            $payload[$key]['store_id'] = $defaultStore;
            $payload[$key]['order_status'] = PurchaseOrder::STATUS_OPEN;
            $payload[$key]['approved_at'] = $now;
            $payload[$key]['purchase_order_number'] = PurchaseOrder::makePurchaseOrderNumber(
                prefix: $prefix,
                sequence: $nextSequence++,
                numOfDigits: $numOfDigits
            );
            $payload[$key]['destination_warehouse_id'] = $warehouseId;
            $payload[$key]['purchase_order_date'] = $now;
            $payload[$key]['currency_id'] = $supplierDefaultCurrencies[$order['supplier_id']]['currency_id'] ?? $defaultCurrency->id;
            $payload[$key]['currency_rate'] = $supplierDefaultCurrencies[$order['supplier_id']]['conversion'] ?? $defaultCurrency->conversion;
            $payload[$key]['currency_id_tenant_snapshot'] = $defaultCurrency->id;

            $orderDto = PurchaseOrderDto::from($payload[$key]);
            $order['purchase_order_lines'] = array_map(function ($line) use ($purchaseOrderNominalCode) {
                $line['nominal_code_id'] = $purchaseOrderNominalCode;
                return $line;
            }, $order['purchase_order_lines']);
            $orderDto->purchaseOrderLines = PurchaseOrderLineDto::collection($order['purchase_order_lines']);
            $data->push($orderDto);
        }

        return PurchaseOrderDto::collection($data->toArray());
    }
}