<?php

namespace App\Services\InventoryForecasting;

use App\Models\Product;
use App\Models\ProductComponent;
use App\Models\SalesOrderLine;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\SupplierProduct;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Inventory\HistoricalData;

/**
 * Trait ForecastsSales.
 */
trait MakesForecastSalesData
{
    /**
     * @throws Exception
     */
    public function makeHistoricalSalesForSku(
        Product $product,
        array $filters = [],
        ?Carbon $startDate = null,
        ?Carbon $endDate = null,
        string $durationUnit = 'month'
    ): HistoricalData {
        $sales = $this->buildHistoricalSales(
            $product,
            $filters,
            $startDate,
            $endDate
        )->get()->map(function (SalesOrderLine $salesOrderLine) {
            return [
                'quantity' => $salesOrderLine->quantity,
                'date' => ( new Carbon($salesOrderLine->salesOrder->order_date) )->format('Y-m-d'),
            ];
        })->toArray();

        return new HistoricalData($durationUnit, $sales);
    }

    public function buildHistoricalSales(
        Product $product,
        array $filters = [],
        ?Carbon $startDate = null,
        ?Carbon $endDate = null
    ): Builder {
        $orderLinesSubQuery = SalesOrderLine::with(['salesOrder']);

        /** @var ProductComponent|null $component */
        $component = ProductComponent::query()->where('component_product_id', $product->id)->first();

        $orderLinesSubQuery->where(function ($q) use ($product, $component) {
            $q->where('product_id', $component?->parent_product_id ?? $product->id);
            if ($component) {
                $q->orWhere('product_id', $component->component_product_id);
            }
        })
            ->whereHas('salesOrder', function (Builder $query) use ($startDate, $endDate, $filters) {
                if ($startDate) {
                    $query->whereDate('sales_orders.order_date', '>=', $startDate);
                }
                if ($endDate) {
                    $query->whereDate('sales_orders.order_date', '<=', $endDate);
                }
                $query->filter($filters);
            });

        $componentsSubQuery = ProductComponent::query()
            ->where('component_product_id', $product->id);

        return SalesOrderLine::with([])
            ->select('sales_order_lines.*', DB::raw('IF(product_components.quantity IS NULL, 1, product_components.quantity) as multiplier'))
            ->fromSub($orderLinesSubQuery, 'sales_order_lines')
            ->leftJoinSub($componentsSubQuery, 'product_components', 'sales_order_lines.product_id', 'product_components.parent_product_id');
    }

    public function buildHistoricalSalesWithProductsQuery(
        Builder $productsQuery,
        array $filters = [],
        ?Carbon $startDate = null,
        ?Carbon $endDate = null
    ): Collection {
        $query = SalesOrderLine::query()
            ->select('products.*',
                DB::raw('coalesce(sum(sales_order_lines.quantity * (IF(product_components.quantity IS NULL, 1, product_components.quantity))), 0) quantity'))
            ->joinSub($productsQuery, 'products', 'products.id', 'sales_order_lines.product_id')
            ->leftJoin('product_components', 'product_components.component_product_id', 'products.id');

        if (empty(@$filters['filters']['filterSet'])) {
            $query->joinRelationship('salesOrder', function ($q) use ($startDate, $endDate) {
                if ($startDate) {
                    $q->whereDate('sales_orders.order_date', '>=', $startDate);
                }
                if ($endDate) {
                    $q->whereDate('sales_orders.order_date', '<=', $endDate);
                }

                return $q;
            });
        } else {
            $query->whereHas('salesOrder', function (Builder $query) use ($startDate, $endDate, $filters) {
                if ($startDate) {
                    $query->whereDate('sales_orders.order_date', '>=', $startDate);
                }
                if ($endDate) {
                    $query->whereDate('sales_orders.order_date', '<=', $endDate);
                }
                $query->filter($filters);
            });
        }

        return $query->groupBy('products.id')->get();
    }

    /**
     * Gets the initial inventory for the given product/sku.
     */
    public function getInitialInventoryForSku(Product $product): float
    {
        // We count the available, in transit (coming in) and
        return $this->getAvailableStock($product) + $this->getInTransitStockForSku($product) + $this->getInboundStockForSku($product);
    }

    /**
     * @return mixed
     */
    public function getInboundStockForSku(Product $product)
    {
        return $product->getIncomingQuantity()->sum('quantity');
    }

    public function getAvailableStock(Product $product): float
    {
        return $product->inWarehouseAvailableQuantity->sum('inventory_available');
    }

    public function getInTransitStockForSku(Product $product): float
    {
        return $product->inWarehouseTransitQuantity->sum('inventory_in_transit');
    }

    public function getProductUnitPrice(Product $product): int
    {
        $defaultPricingTier = $product->productPricingTiers()->where('is_default', 1)->first();

        return $defaultPricingTier->pivot->price ?? 0;
    }

    public function getDefaultProductSupplierPrice(Supplier $supplier, Product $product): float
    {
        /** @var SupplierProduct $supplierProduct */
        $supplierProduct = $product->supplierProducts()
            ->where('supplier_id', $supplier->id)
            ->firstOrFail();

        if (! ($pricingTierId = $supplier->default_pricing_tier_id)) {
            // We get the global pricing tier id in settings
            $pricingTierId = SupplierPricingTier::with([])->where('is_default', 1)->firstOrFail()->id;
        }

        $pricing = $supplierProduct
            ->supplierProductPricing()
            ->where('supplier_pricing_tier_id', $pricingTierId)->first();

        return $pricing->price ?? ($product->average_cost ?? 0.0);
    }
}
