<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 10/17/20
 * Time: 9:57 PM.
 */

namespace App\Services\InventoryForecasting;

use App\Helpers;
use App\Jobs\ForecastSupplierInventoryJob;
use App\Models\Currency;
use App\Models\InventoryForecast;
use App\Models\Product;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Supplier;
use App\Repositories\InventoryForecastRepository;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Builders\SalesBasedPurchaseOrderBuilder;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Exceptions\LargePurchaseOrderException;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilderLine;
use Carbon\Carbon;
use Exception;
use Illuminate\Bus\Batch;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Redis;
use Inventory\HistoricalData;
use Throwable;

/**
 * Class ForecastManager.
 */
class ForecastManager
{
    use MakesForecastSalesData;

    /**
     * Monthly Analysis.
     */
    const DURATION_UNIT_MONTH = 'month';
    const INVENTORY_FORECASTING_BATCH = 'INVENTORY_FORECASTING_BATCH';

    private bool $adjustForecast = true;

    /**
     * Zero means no limit.
     */
    private int $salesHistoryInDays = 90;

    private int $maxProjectionDays = 90;

    private InventoryForecastRepository $forecasts;

    /**
     * ForecastManager constructor.
     */
    public function __construct(InventoryForecastRepository $forecasts)
    {
        $this->forecasts = $forecasts;
        $this->initializeSettings();
    }

    /**
     * @param int|array $productIds
     * @return void
     * @throws Throwable
     */
    public function forecast(int|array $productIds = []): void
    {

        // We clear prior forecast if not forecasting for specific products
        $this->forecasts->clear();

        $jobs = [];

        $suppliersQuery = Supplier::query()
            ->whereHas('products', function($q){
                $q->where('type', '!=', Product::TYPE_KIT)
                    ->where('is_default', true);
            });

        $total = $suppliersQuery->count();
        if($total === 0){
            return;
        }


        $suppliersQuery
            ->chunk(500, function($suppliers) use (&$jobs) {
                foreach($suppliers as $supplier){

                    try {
                        // Perform the forecast for the supplier as a job
                        $jobs[] = new ForecastSupplierInventoryJob(
                            manager: $this,
                            cache: $this->forecasts,
                            supplier: $supplier,
                        );
                    }
                    catch (Exception $e) {
                        // We continue with the next supplier
                        slack('Forecasting failed for supplier ' . $supplier->name . ' with message: ' . $e->getMessage());
                    }

                }
            });


        $this->initiateForecasting(count($jobs));
        // Dispatch the jobs
        Bus::batch($jobs)
            ->progress(function(Batch $batch){
                // Update progress
                $tracking = Redis::get(self::INVENTORY_FORECASTING_BATCH);
                $tracking = json_decode($tracking, true);
                $tracking['progress'] = $batch->progress();
                Redis::set(self::INVENTORY_FORECASTING_BATCH, json_encode($tracking));
            })
            ->finally(function(Batch $batch){
                // Handle completion
                $tracking = Redis::get(self::INVENTORY_FORECASTING_BATCH);
                $tracking = json_decode($tracking, true);
                $tracking['completed_at'] = now(Helpers::getAppTimezone() ?: 'UTC')->format('Y-m-d H:i:s');
                $tracking['progress'] = $batch->progress();
                Redis::set(self::INVENTORY_FORECASTING_BATCH, json_encode($tracking));
            })
            ->onQueue('forecasting')
            ->name('Inventory Forecasting')
            ->dispatch();
    }

    private function initiateForecasting(int $total): void
    {
        Redis::set(self::INVENTORY_FORECASTING_BATCH, json_encode([
            'total' => $total,
            'progress' => 0,
            'started_at' => now(Helpers::getAppTimezone() ?: 'UTC')->format('Y-m-d H:i:s'),
            'completed_at' => null,
            'days_sales_history' => $this->salesHistoryInDays,
            'max_projection_days' => $this->maxProjectionDays,
        ]));
    }

    /**
     * @throws BindingResolutionException
     * @throws LargePurchaseOrderException
     */
    public function forecastForSupplier(
        Supplier $supplier,
        bool $forSpecificProducts = false,
        array $productIds = []
    ): void
    {
        $productFilters = [];
        if ($forSpecificProducts) {
            $productFilters = [
                'filters' => [
                    'conjunction' => 'and',
                    'filterSet' => [
                        [
                            'column' => 'id',
                            'operator' => 'isAnyOf',
                            'value' => implode(',', $productIds),
                        ],
                    ],
                ],
            ];
        }

        $timezone = Helpers::getAppTimezone() ?: 'UTC';

        $builder = new SalesBasedPurchaseOrderBuilder(
            supplier: $supplier,
            options: [
                'sales_start_date' => now($timezone)->subDays($this->salesHistoryInDays),
                'sales_end_date' => now($timezone),
                'target_stock_days' => $this->maxProjectionDays,
                'use_leadtime' => true,
                'use_moq' => true,
                'product_filters' => $productFilters,
            ]
        );

        $data = $builder->build();

        if($data->isEmpty()){
            return;
        }

        $forecast = $this->forecasts->createForecast(
            supplierId: $supplier->id,
            daysOfSalesHistory: $this->salesHistoryInDays,
            maxProjectionDays: $this->maxProjectionDays,
            leadtime: $calculation['leadtime'] ?? 0,
            minimumOrderQuantity: $supplier->minimum_order_quantity ?? 0,
        );

        $results = $data->map(function (PurchaseOrderBuilderLine $line) use ($timezone, $forecast) {
            $supplier = $line->getSupplierProduct()->supplier;
            $product = $line->getSupplierProduct()->product;
            $calculation = $line->getCalculation();

            if ($calculation['daily_average_consumption'] <= 0) {
                $numOfDays = 0;
            } else {
                $stock = $calculation['num_stock'] + // Current stock which is negative if backordered
                    $calculation['num_inbound'] + $calculation['num_in_transit'];
                $numOfDays = ceil(
                    max(0, $stock) / $calculation['daily_average_consumption']
                );
            }

            $outOfStockDate = now($timezone)->addDays($numOfDays)->format('Y-m-d');

            // Cache default currency and supplier currency conversion rate.
            $tenantCurrency = Currency::default();
            if($supplier->defaultPricingTier){
                $supplierDefaultCurrency = Currency::query()
                    ->where('code', $supplier->defaultPricingTier->currency_code)->firstOrFail();
            }

            $unitCost = $this->getDefaultProductSupplierPrice($supplier, $product);

            return [
                'inventory_forecast_id' => $forecast->id,
                'product_id' => $product->id,
                'warehouse_id' => $line->getWarehouseId(),
                'suggested_purchase_quantity' => $line->getQuantity(),
                'out_of_stock_date' => $outOfStockDate,
                'unit_cost' => $unitCost,
                'tenant_currency_id' => $tenantCurrency->id,
                'tenant_currency_code' => $tenantCurrency->code,
                'unit_cost_in_tenant_currency' => ($supplierDefaultCurrency ?? $tenantCurrency)->conversion * $unitCost,
                'inbound_stock' => $calculation['num_inbound'],
                'available_stock' => $calculation['num_stock'],
                'in_transit_stock' => $calculation['num_in_transit'],
            ];
        })->toArray();

        // Record the results in the cache.
        $this->forecasts->recordAll($results);

    }

    /**
     * @throws Exception
     */
    public function makeHistoricalSalesForSku(Product $product): HistoricalData
    {
        // We use the order reports as historical sales data for the product
        $query = SalesOrderLine::with([])
            ->where('product_id', $product->id);

        if ($this->salesHistoryInDays != 0) {
            // There is a limit for sales history
            $dateLimit =
            $query->whereHas('salesOrder', function (Builder $query) {
                $query->whereDate(
                    'order_date',
                    '>=',
                    Carbon::now()->subDays($this->salesHistoryInDays)
                );
            });
        }

        $sales = $query->get()->map(function (SalesOrderLine $salesOrderLine) {
            return [
                'quantity' => $salesOrderLine->quantity,
                'date' => ( new Carbon($salesOrderLine->salesOrder->order_date) )->format('Y-m-d'),
            ];
        })->toArray();

        return new HistoricalData(self::DURATION_UNIT_MONTH, $sales);
    }

    private function initializeSettings(): void
    {
        $setting = Setting::with([])->whereIn('key', Setting::KEYS_INVENTORY_FORECASTING)->get();
        $this->adjustForecast = $setting->where('key', Setting::KEY_ADJUST_FORECAST_OUT_OF_STOCK)->first()->value ?? true;
        $this->salesHistoryInDays = $setting->where('key', Setting::KEY_DAYS_SALES_HISTORY)->first()->value ?? $this->salesHistoryInDays;
        $this->maxProjectionDays = $setting->where('key', Setting::KEY_MAX_PROJECTION_DAYS)->first()->value ?? $this->maxProjectionDays;
    }
}
