<?php

namespace App\Services\InventoryAdjustment;

use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;

class UpdateInventoryAdjustmentService extends InventoryAdjustmentService
{
    /** @var InventoryAdjustment */
    private $adjustment;

    public function execute(?InventoryAdjustment $inventoryAdjustment = null): ?InventoryAdjustment
    {
        if (! $inventoryAdjustment) {
            throw new InvalidArgumentException('must send the Inventory Adjustment that you want to update.');
        }

        $this->adjustment = $inventoryAdjustment;

        DB::transaction(function () {
            /**
             * update adjustment, inventory movements and fifo-layer data except quantity attribute
             * maybe some movements did not update when updating quantity.
             */
            $this->updateInventoryMovements();
            $this->updateFifoLayer();
            $this->adjustment->update(Arr::except($this->inventoryAdjustmentRequest->validated(), ['quantity']));

            // update inventory adjustment quantity will affect on movements and fifo-layers
            // TODO:: Check this
            $this->updateAdjustmentQuantityAndCost();
        });

        return $this->adjustment;
    }

    /**
     * @throws Exception
     */
    private function enforceUpdateChecks()
    {
        $data = $this->inventoryAdjustmentRequest->validated();

        // These checks are only applicable to changing the quantity of
        // the adjustment
        if (! isset($data['quantity'])) {
            return;
        }

        $quantity = $data['quantity'];

        // For positive adjustments, updates can only happen if the original
        // fifo layer created isn't used
        if ($this->adjustment->quantity > 0) {
            $usedLayersCount = $this->adjustment->fifoLayers()->where('fulfilled_quantity', '!=', 0)->count();
            if ($usedLayersCount > 0) {
                throw new Exception('FIFO Layer is already used from this adjustment, updates are disallowed.');
            }
        } else {
            // For updates to negative adjustments,
            // updates are disallowed if there have been
            // inventory movements after the adjustment was made.
            /** @var InventoryMovement $adjustmentMovement */
            $adjustmentMovement = $this->adjustment->inventoryMovements()->firstOrFail();
            $moreMovementsCount = InventoryMovement::with([])->whereDate('inventory_movement_date', '>', $adjustmentMovement->inventory_movement_date)->count();
            if ($moreMovementsCount > 0) {
                throw new Exception('There are other inventory movements after this adjustment, updates are disallowed.');
            }
        }
    }

    /**
     * Update Inventory adjustment quantity.
     *
     * @throws \App\Exceptions\InsufficientStockException
     * @throws Exception
     */
    private function updateAdjustmentQuantityAndCost()
    {
        $inputs = $this->inventoryAdjustmentRequest->validated();

        $adjustedQuantity = $this->getQuantity($this->adjustment);

        if (isset($inputs['quantity']) && $adjustedQuantity !== $this->adjustment->quantity) {
            if ($this->adjustment->quantity > 0) { // adjustment was positive(increase)
                PositiveInventoryAdjustmentService::make($this->inventoryAdjustmentRequest)->update($this->adjustment);
            } else { // adjustment was negative(decrease)
                NegativeInventoryAdjustmentService::make($this->inventoryAdjustmentRequest)->update($this->adjustment);
            }

        // Update adjustment quantity
        } elseif (isset($inputs['unit_cost']) && $inputs['unit_cost'] !== $this->adjustment->unit_cost) {
            // if the quantity changed, it will handle unit_cost changes
            // this changes when unit_cost changed without quantity
            $this->adjustment->fifoLayers()->each(function (FifoLayer $fifoLayer) use ($inputs) {
                $fifoLayer->total_cost = $fifoLayer->original_quantity * $inputs['unit_cost'];
                $fifoLayer->save();
            });
        }
    }

    /**
     * Update inventory movements of the inventory adjustment.
     *
     * update: warehouse_id, warehouse_location_id and date
     */
    private function updateInventoryMovements()
    {
        $inputs = $this->inventoryAdjustmentRequest->validated();
        $data = [];

        // update warehouse
        if (isset($inputs['warehouse_id']) && $inputs['warehouse_id'] !== $this->adjustment->warehouse_id) {
            $data['warehouse_id'] = $inputs['warehouse_id'];
            $data['warehouse_location_id'] = $inputs['warehouse_location_id'] ?? null;
        }

        // update date
        if (isset($inputs['adjustment_date']) && ! $this->adjustment->adjustment_date->isSameDay($inputs['adjustment_date'])) {
            $data['inventory_movement_date'] = $inputs['adjustment_date'];
        }

        // update inventory movements
        if (! empty($data)) {
            // must update separately to fire events
            $this->adjustment->inventoryMovements()->each(function (InventoryMovement $inventoryMovement) use ($data) {
                $inventoryMovement->update($data);
            });
        }
    }

    /**
     * Update fifo-layer for the positive inventory adjustment.
     */
    private function updateFifoLayer()
    {
        $inputs = $this->inventoryAdjustmentRequest->validated();
        $data = [];

        // update date
        if (isset($inputs['adjustment_date']) && ! $this->adjustment->adjustment_date->isSameDay($inputs['adjustment_date'])) {
            $data['fifo_layer_date'] = Carbon::parse($inputs['adjustment_date'])->timezone(config('app.timezone'));
        }

        $unitCost = $inputs['unit_cost'] ?? $this->adjustment->unit_cost;
        $quantity = $inputs['quantity'] ?? $this->adjustment->quantity;
        $data['total_cost'] = $unitCost * $quantity;

        if (! empty($data)) {
            $this->adjustment->fifoLayers()->update($data);
        }
    }
}
