<?php

namespace App\Services\InventoryAdjustment;

use App\Exceptions\InsufficientStockException;
use App\Models\InventoryAdjustment;
use App\Services\InventoryManagement\InventoryManager;
use Illuminate\Support\Facades\DB;
use Throwable;

class PositiveInventoryAdjustmentService extends InventoryAdjustmentService
{
    /**
     * Add positive inventory adjustment
     * this function only calls when you want to create a new positive inventory adjustment OR
     * convert a negative inventory adjustment to positive adjustment.
     *
     *
     * @throws Throwable
     */
    public function execute(?InventoryAdjustment $inventoryAdjustment = null): ?InventoryAdjustment
    {
        return DB::transaction(function () use ($inventoryAdjustment) {
            if (! $inventoryAdjustment) {
                $inventoryAdjustment = $this->createInventoryAdjustment();
            }

            InventoryManager::with(
                $inventoryAdjustment->warehouse_id,
                $inventoryAdjustment->product
            )->addToStock(abs($inventoryAdjustment->quantity), $inventoryAdjustment, unitCost: $this->inventoryAdjustmentRequest->unit_cost);

            return $inventoryAdjustment;
        });
    }

    /**
     * Update Positive inventory adjustment.
     *
     *
     * @throws InsufficientStockException
     */
    public function update(InventoryAdjustment $inventoryAdjustment): ?InventoryAdjustment
    {
        $manager = InventoryManager::with(
            $inventoryAdjustment->warehouse_id,
            $inventoryAdjustment->product
        );

        $adjustedQuantity = $this->getQuantity($inventoryAdjustment);

        if ($adjustedQuantity > 0) {
            $diff = $adjustedQuantity - $inventoryAdjustment->quantity;
            if ($diff > 0) {
                $manager->increasePositiveEventQty($diff, $inventoryAdjustment);
            } else {
                $manager->reducePositiveEventQty(abs($diff), $inventoryAdjustment);
            }
        } elseif ($adjustedQuantity < 0) {
            /**
             * The user is reversing an increase in inventory to a
             * decrease in inventory. We reverse the inventory on the
             * positive adjustment and create a negative inventory adjustment
             * event.
             */
            $manager->removeAllStockFrom($inventoryAdjustment);
            $manager->takeFromStock(abs($adjustedQuantity), $inventoryAdjustment);
        }

        $inventoryAdjustment->quantity = $adjustedQuantity;
        $inventoryAdjustment->save();

        return $inventoryAdjustment;
    }
}
