<?php

namespace App\Services\InventoryAdjustment;

use App\Exceptions\InsufficientStockException;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Services\InventoryManagement\BackorderManager;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\InventoryManagement\InventoryReductionCause;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;

/**
 * Class NegativeInventoryAdjustmentService
 */
class NegativeInventoryAdjustmentService extends InventoryAdjustmentService
{
    /**
     * Holds fifo layers involved in the release of reservations
     */
    protected static array $releasedFifoLayers = [];

    public function execute(?InventoryAdjustment $inventoryAdjustment = null): ?InventoryAdjustment
    {
        return DB::transaction(function () use ($inventoryAdjustment) {
            if (! $inventoryAdjustment) {
                $inventoryAdjustment = $this->createInventoryAdjustment();
            }

            InventoryManager::with(
                $inventoryAdjustment->warehouse_id,
                $inventoryAdjustment->product
            )->takeFromStock(abs($inventoryAdjustment->quantity), $inventoryAdjustment);

            return $inventoryAdjustment;
        });
    }

    /**
     * @throws InsufficientStockException
     * @throws \Exception
     */
    public function update(InventoryAdjustment $inventoryAdjustment): ?InventoryAdjustment
    {
        $manager = InventoryManager::with(
            $inventoryAdjustment->warehouse_id,
            $inventoryAdjustment->product
        );

        $adjustedQuantity = $this->getQuantity($inventoryAdjustment);
        if ($adjustedQuantity < 0) {
            $diff = $adjustedQuantity - $inventoryAdjustment->quantity;
            if ($diff < 0) {
                // Increasing a negative adjustment qty.
                $manager->increaseNegativeEventQty(abs($diff), $inventoryAdjustment);
            } elseif ($diff > 0) {
                // Decrease in negative adjustment qty.
                $manager->decreaseNegativeEventQty(abs($diff), $inventoryAdjustment);
            }
        } elseif ($adjustedQuantity > 0) {
            /**
             * The user is changing a negative adjustment to a positive one.
             * We give back the inventory on the original negative adjustment
             * and add the new quantity to stock.
             */
            $manager->reverseNegativeEvent($inventoryAdjustment);
            $manager->addToStock($adjustedQuantity, $inventoryAdjustment, unitCost: $this->inventoryAdjustmentRequest->unit_cost);
        }

        $inventoryAdjustment->quantity = $adjustedQuantity;
        $inventoryAdjustment->save();

        return $inventoryAdjustment;
    }

    public static function allQuantityAppliedToSalesOrders(int $productId, int $warehouseId, int $quantity): bool
    {
        $appliedQuantity = 0;

        SalesOrder::with(['salesOrderLines', 'salesOrderLines.inventoryMovements'])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderLines', function (Builder $builder) use ($productId, $warehouseId) {
                return $builder->where('product_id', $productId)
                    ->where('warehouse_id', $warehouseId);
            })->each(function (SalesOrder $salesOrder) use (&$appliedQuantity, $quantity, $productId, $warehouseId) {
                // We end whenever we're done applying all the needed quantity.
                if ($appliedQuantity == $quantity) {
                    return;
                }

                $appliedQuantity += self::applyQuantityToSalesOrder($salesOrder, $quantity, $productId, $warehouseId);
            });

        return $appliedQuantity == $quantity;
    }

    private static function applyQuantityToSalesOrder(SalesOrder $salesOrder, int $quantity, int $productId, int $warehouseId): int
    {
        $usedQuantity = 0;

        /** @var BackorderManager $backorderManager */
        $backorderManager = app(BackorderManager::class);

        $salesOrder->salesOrderLines()
            ->with(['backorderQueue', 'salesOrderFulfillmentLines'])
            ->where('product_id', $productId)
            ->where('warehouse_id', $warehouseId)
            ->each(function (SalesOrderLine $orderLine) use (&$usedQuantity, $quantity, $backorderManager) {
                if ($usedQuantity >= $quantity) {
                    return;
                }

                if (! ($releaseableQuantity = $orderLine->inventoryMovements()->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)->sum('quantity'))) {
                    // The order line is already fulfilled and can't release quantities.
                    return;
                }

                /**
                 * The quantity releaseable for this line must not include backordered quantity for the line.
                 * Neither must it not exceed the overall quantity remaining to be released.
                 */
                $releaseableQuantity = max(0, min($releaseableQuantity - $orderLine->backordered_quantity - $orderLine->fulfilled_quantity, ($quantity - $usedQuantity)));
                if ($releaseableQuantity == 0) {
                    return; // There are no more quantities to release for this sales order line.
                }

                /**
                 * We have a quantity to release for the line,
                 * we need to iteratively swap layers (fifo -> backorder queue)
                 * for the total quantity releasable
                 */
                $processedFifoLayerIds = [];
                do {
                    /** @var SalesOrderLineLayer $lineLayer */
                    $lineLayer = SalesOrderLineLayer::with([])
                        ->where('sales_order_line_id', $orderLine->id)
                        ->where('layer_type', FifoLayer::class)
                        ->whereNotIn('layer_id', $processedFifoLayerIds)
                        ->first();

                    if (! $lineLayer) {
                        break;
                    }

                    /** @var FifoLayer $fifoLayer */
                    $fifoLayer = $lineLayer->layer;

                    if ($lineLayer->quantity < $releaseableQuantity) {
                        /**
                         * We keep track of the fifo layer to be used
                         * for the inventory adjustment. This is later used
                         * to ensure that the exact fifo released is what's used
                         * for the inventory adjustment. Note that the whole goal
                         * of this is to move fifo from sales orders for the negative adjustment.
                         */
                        self::$releasedFifoLayers[] = [
                            'layer' => $fifoLayer,
                            'quantity' => $lineLayer->quantity,
                        ];

                        // We create the backorder queue
                        $backorderQueue = $backorderManager->createBackorderQueue(
                            $lineLayer->quantity,
                            $orderLine->id
                        );

                        /**
                         * Next, we update the inventory movements to use the backorder layer
                         * not the fifo layer.
                         */
                        self::moveOrderLineMovementsToBackorder($orderLine, $fifoLayer, $backorderQueue);

                        // Update the releasable quantity
                        $releaseableQuantity -= $lineLayer->quantity;

                        // Update the used quantity
                        $usedQuantity += $lineLayer->quantity;

                        /**
                         * Since the fifo quantity has been put back and a backorder has been created
                         * for the line, we no longer need the line layer
                         */
                        SalesOrderLineLayer::with([])->where('id', $lineLayer->id)->delete();
                    } else {
                        /**
                         * Again, we keep track of the fifo layer
                         * for the creation and mapping of the negative
                         * adjustment.
                         */
                        self::$releasedFifoLayers[] = [
                            'layer' => $fifoLayer,
                            'quantity' => $releaseableQuantity,
                        ];

                        // We create the backorder queue
                        $backorderQueue = $backorderManager->createBackorderQueue(
                            $releaseableQuantity,
                            $orderLine->id,
                            InventoryReductionCause::INCREASED_NEGATIVE_EVENT,
                            $orderLine->product->defaultSupplierProduct ? $orderLine->product->defaultSupplierProduct->supplier_id : null,
                        );

                        /**
                         * Next, we update the inventory movements to use the backorder layer
                         * not the fifo layer.
                         */
                        self::moveOrderLineMovementsToBackorder($orderLine, $fifoLayer, $backorderQueue, $releaseableQuantity);

                        /**
                         * We update the line layer. Thus, we delete it if all the quantity is moved or
                         * update it's quantity if the move was a partial move.
                         */
                        if ($lineLayer->quantity == $releaseableQuantity) {
                            /**
                             * All the quantity is released, no need for the line layer
                             * at this point.
                             */
                            SalesOrderLineLayer::with([])->where('id', $lineLayer->id)->delete();
                        } else {
                            /**
                             * The quantity on the line layer is more than what's needed
                             * for the release, we simply adjust the line layer quantity
                             * to reflect the balance.
                             */
                            $lineLayer->quantity -= $releaseableQuantity;
                            $lineLayer->save();
                        }

                        // Update used quantity
                        $usedQuantity += $releaseableQuantity;

                        // All releaseable quantity is released.
                        $releaseableQuantity = 0;
                    }

                    // We mark the fifo layer as processed.
                    $processedFifoLayerIds[] = $fifoLayer->id;
                } while ($releaseableQuantity <= 0);
            });

        return $usedQuantity;
    }

    /**
     * @throws OpenStockTakeException
     */
    protected static function moveOrderLineMovementsToBackorder(
        SalesOrderLine $orderLine,
        FifoLayer $fifoLayer,
        BackorderQueue $backorderQueue,
        ?int $quantityToMove = null
    ) {
        if (is_null($quantityToMove)) {
            /**
             * No quantity is specified, we'll take it to
             * mean that we need to move the entire quantity
             * from the fifo layer to the backorder queue.
             */
            $orderLine->inventoryMovements()
                ->where('layer_type', FifoLayer::class)
                ->where('layer_id', $fifoLayer->id)
                ->where(function (Builder $builder) {
                    return $builder->where(function (Builder $builder) {
                        return $builder->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                            ->where('quantity', '>', 0);
                    })
                        ->OrWhere(function (Builder $builder) {
                            return $builder->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                                ->where('quantity', '<', 0);
                        });
                })
                ->update(['layer_type' => BackorderQueue::class, 'layer_id' => $backorderQueue->id]);
        } else {
            /**
             * We balance the reserved and active inventory
             */
            /** @var InventoryMovement $reservedMovement */
            $reservedMovement = $orderLine->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                ->where('quantity', '>', 0)
                ->firstOrFail();

            /** @var InventoryMovement $activeMovement */
            $activeMovement = $orderLine->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('quantity', '<', 0)
                ->firstOrFail();

            $movementQuantity = $reservedMovement->quantity;
            $reservedMovement->quantity = max(0, $movementQuantity - $quantityToMove);
            if ($reservedMovement->quantity == 0) {
                /**
                 * Looks like the entire quantity is being moved, we simply map
                 * the movement to the backorder queue for the original quantity
                 */
                $reservedMovement->quantity = $movementQuantity;
                $reservedMovement->backorder_queue = $backorderQueue->id;
                $reservedMovement->save();

                $activeMovement->quantity = -$movementQuantity;
                $activeMovement->backorder_queue = $backorderQueue->id;
                $activeMovement->save();
            } else {
                /**
                 * We update the reserved and active movements
                 */
                $reservedMovement->save();

                $activeMovement->update(['quantity' => -$reservedMovement->quantity]);

                /**
                 * Finally, we create the split movements for the backorder queue
                 */
                $reservedMovement = $reservedMovement->replicate();
                $reservedMovement->quantity = $quantityToMove;
                $reservedMovement->backorder_queue = $backorderQueue->id;
                $reservedMovement->save();

                $activeMovement = $activeMovement->replicate();
                $activeMovement->quantity = -$quantityToMove;
                $activeMovement->backorder_queue = $backorderQueue->id;
                $activeMovement->save();
            }
        }
    }
}
