<?php

namespace App\Services\InventoryAdjustment;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use Exception;
use Illuminate\Database\Eloquent\Model;
use InvalidArgumentException;

/**
 * @deprecated Use InventoryAdjustmentManager instead
 */
abstract class InventoryAdjustmentService
{
    protected $inventoryAdjustmentRequest;

    public function __construct(StoreInventoryAdjustment $request)
    {
        $this->inventoryAdjustmentRequest = $request;
    }

    public static function make(StoreInventoryAdjustment $request)
    {
        return new static($request);
    }

    abstract public function execute(?InventoryAdjustment $inventoryAdjustment = null): ?InventoryAdjustment;

    /**
     * Action factory.
     *
     *
     * @return InventoryAdjustmentService
     *
     * @throws Exception
     */
    public static function factory(StoreInventoryAdjustment $request)
    {
        switch ($request->input('adjustment_type')) {
            case StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE:
                return PositiveInventoryAdjustmentService::make($request);
            case StoreInventoryAdjustment::ADJUSTMENT_TYPE_DECREASE:
                return NegativeInventoryAdjustmentService::make($request);
            case StoreInventoryAdjustment::ADJUSTMENT_TYPE_SET:
                return SetInventoryAdjustmentService::make($request);
            default:
                throw new InvalidArgumentException('Invalid adjustment type. valid values:'.implode(',', StoreInventoryAdjustment::ADJUSTMENT_TYPES));
        }
    }

    /**
     * Get adjustment quantity depends on type.
     */
    protected function getQuantity(?InventoryAdjustment $inventoryAdjustment = null): float
    {
        $adjustmentType = $this->inventoryAdjustmentRequest->input('adjustment_type');
        if (! $adjustmentType) {
            $adjustmentType = $inventoryAdjustment->quantity > 0 ? StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE : StoreInventoryAdjustment::ADJUSTMENT_TYPE_DECREASE;
        }

        return match ($adjustmentType) {
            StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE, StoreInventoryAdjustment::ADJUSTMENT_TYPE_SET => abs($this->inventoryAdjustmentRequest->input('quantity')),
            StoreInventoryAdjustment::ADJUSTMENT_TYPE_DECREASE => -abs($this->inventoryAdjustmentRequest->input('quantity')),
            default => $this->inventoryAdjustmentRequest->input('quantity'),
        };
    }

    /**
     * Create Inventory Adjustment from request.
     *
     * @return InventoryAdjustment|Model
     */
    protected function createInventoryAdjustment()
    {
        $data = $this->inventoryAdjustmentRequest->validated();

        $data['quantity'] = $this->getQuantity();

        return InventoryAdjustment::with([])->create($data);
    }

    /**
     * Update Inventory Adjustment from request.
     *
     *
     * @return bool
     */
    protected function updateInventoryAdjustment(InventoryAdjustment $inventoryAdjustment)
    {
        $data = $this->inventoryAdjustmentRequest->validated();

        if (isset($data['quantity'])) {
            $data['quantity'] = $this->getQuantity($inventoryAdjustment);
        }

        return $inventoryAdjustment->update($data);
    }

    /**
     * Create Inventory Movement from inventory adjustment.
     */
    protected function addInventoryMovement(InventoryAdjustment $adjustment, FifoLayer $fifoLayer, ?int $quantity = null, ?InventoryMovement $inventoryMovement = null): InventoryMovement
    {
        if (! $inventoryMovement) {
            $inventoryMovement = new InventoryMovement();
        }

        $inventoryMovement->inventory_movement_date = $adjustment->adjustment_date;
        $inventoryMovement->product_id = $adjustment->product_id;
        $inventoryMovement->quantity = $quantity ?: $adjustment->quantity;
        $inventoryMovement->type = InventoryMovement::TYPE_ADJUSTMENT;
        $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $inventoryMovement->warehouse_id = $adjustment->warehouse_id;
        $inventoryMovement->warehouse_location_id = $this->inventoryAdjustmentRequest->input('warehouse_location_id');
        $inventoryMovement->fifo_layer = $fifoLayer->id;

        $adjustment->inventoryMovements()->save($inventoryMovement);

        return $inventoryMovement;
    }
}
