<?php

namespace App\Services\FinancialManagement;

use App\DTO\ReportingDailyFinancialDto;
use App\Helpers;
use App\Jobs\GenerateCacheDailyAverageConsumptionForProductsJob;
use App\Jobs\CalculateDailyFinancialsJob;
use App\Models\FinancialLineType;
use App\Models\Product;
use App\Models\ReportingDailyFinancial;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Repositories\DailyFinancialRepository;
use App\Repositories\FinancialLineTypeRepository;
use App\Repositories\InventorySnapshotRepository;
use App\Repositories\ProductRepository;
use App\Repositories\SalesOrderLineRepository;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\LazyCollection;

class DailyFinancialManager
{
    private DailyFinancialRepository $dailyFinancialRepository;

    private ProductRepository $productRepository;

    private SalesOrderLineRepository $salesOrderLineRepository;

    private InventorySnapshotRepository $inventorySnapshotRepository;

    private FinancialLineTypeRepository $financialLineTypeRepository;

    private ?string $userTimezone;

    public function __construct()
    {
        $this->dailyFinancialRepository = app(DailyFinancialRepository::class);
        $this->productRepository = app(ProductRepository::class);
        $this->salesOrderLineRepository = app(SalesOrderLineRepository::class);
        $this->inventorySnapshotRepository = app(InventorySnapshotRepository::class);
        $this->financialLineTypeRepository = app(FinancialLineTypeRepository::class);
        $this->userTimezone = Helpers::getAppTimezone();
    }

    public function calculate(array $days = []): void
    {
        //$this->dailyFinancialRepository->purgeInvalidDailyFinancials($days);
        $this->seedDailyFinancials();

        if(empty($days)) {
            $days = $this->dailyFinancialRepository->getInvalidatedDays();
        }

        foreach (array_chunk($days, 10) as $chunk_days) {
            customlog('daily_financials', 'Get invalidated daily financials', [
                'start' => $chunk_days[0],
                'end' => $chunk_days[count($chunk_days) - 1],
            ]);
            $invalidatedDailyFinancials = $this->dailyFinancialRepository->getInvalidatedDailyFinancials($chunk_days);

            customlog('daily_financials', 'Invalidated daily financials count: '.$invalidatedDailyFinancials->count());
            if (! empty($invalidatedDailyFinancials)) {
                customlog('daily_financials', 'Calculate and cache daily financials');
                $this->calculateAndCacheDailyFinancials($invalidatedDailyFinancials);
                customlog('daily_financials', 'Calculate and cache daily financials done');
            }

            customlog('daily_financials', 'Clear invalid daily financials cache');
            $this->dailyFinancialRepository->clearInvalidDailyFinancialsCache($chunk_days);
        }
        $this->dailyFinancialRepository->deleteEmptyDailyFinancials();
    }

    public function calculateAndCacheDailyFinancials(LazyCollection $reportingDailyFinancials): void
    {
        if ($reportingDailyFinancials->count() == 0) {
            return;
        }

        $dailyAverageConsumptionStartDate = Carbon::parse(now()->subDays(Helpers::setting(Setting::KEY_DAYS_SALES_HISTORY, 90)));
        $productIdsToInvalidateDailyAverageConsumptionCache = [];

        $reportingDailyFinancialCollection = new Collection();

        $i = 0;
        $reportingDailyFinancials->each(function (ReportingDailyFinancial $reportingDailyFinancial) use (&$i, $reportingDailyFinancialCollection, $dailyAverageConsumptionStartDate, &$productIdsToInvalidateDailyAverageConsumptionCache) {
            $i++;
            if ($i % 1000 == 0) {
                customlog('daily_financials', 'Processed '.$i.' daily financials');
            }

            if (! $reportingDailyFinancial->reportable) {
                customlog('daily_financials', 'Reportable link not active, skipping');
                return;
            }
            $data = $reportingDailyFinancial->reportable->getReportingDailyFinancialDto($reportingDailyFinancial->date);
            if ($data) {
                $reportingDailyFinancialCollection->add($data);
            }

            /*
             * Handle cache invalidations
             */
            if ($reportingDailyFinancial->reportable_type == Product::class) {
                /*
                 * Build an array of products with reporting updates so that we know to invalidate the daily average
                 * consumption cache
                 */
                if ($reportingDailyFinancial->date >= $dailyAverageConsumptionStartDate) {
                    $productIdsToInvalidateDailyAverageConsumptionCache[] = $reportingDailyFinancial->reportable_id;
                }
            }
        });

        customlog('daily_financials', 'Saving bulk');
        $this->dailyFinancialRepository->save($reportingDailyFinancialCollection, ReportingDailyFinancial::class);
        customlog('daily_financials', 'Saving bulk done');

        // Invalidate daily average consumption cache for all given products
        $productIdsToInvalidateDailyAverageConsumptionCache = array_unique($productIdsToInvalidateDailyAverageConsumptionCache);
        customlog('daily_financials', 'Invalidating Daily Average Consumption Cache');
        $this->productRepository->invalidateDailyAverageConsumptionCache($productIdsToInvalidateDailyAverageConsumptionCache);

        customlog('daily_financials', 'Dispatching CacheDailyAverageConsumptionForProductsJob');
        // Because we may have new invalidated records, we want to run the jobs to validate them
        dispatch(new GenerateCacheDailyAverageConsumptionForProductsJob())->onQueue('products');
    }

    public function recalculateDailyFinancials(Collection $dailyFinancials): void
    {
        $dailyAverageConsumptionStartDate = Carbon::parse(now()->subDays(Helpers::setting(Setting::KEY_DAYS_SALES_HISTORY, 90)));
        $reportingDailyFinancialCollection = new Collection();
        $productIdsToInvalidateDailyAverageConsumptionCache = [];
        $dailyFinancials->each(function (ReportingDailyFinancial $dailyFinancial) use ($reportingDailyFinancialCollection, $dailyAverageConsumptionStartDate, &$productIdsToInvalidateDailyAverageConsumptionCache) {
            $data = $dailyFinancial->reportable->getReportingDailyFinancialDto($dailyFinancial->date);
            if ($data) {
                $reportingDailyFinancialCollection->add($data);
            }
            if ($dailyFinancial->reportable_type == Product::class) {
                if ($dailyFinancial->date >= $dailyAverageConsumptionStartDate) {
                    $productIdsToInvalidateDailyAverageConsumptionCache[] = $dailyFinancial->reportable_id;
                }
            }
        });

        $this->dailyFinancialRepository->save($reportingDailyFinancialCollection, ReportingDailyFinancial::class);
        $this->dailyFinancialRepository->deleteEmptyDailyFinancials();
        $productIdsToInvalidateDailyAverageConsumptionCache = array_unique($productIdsToInvalidateDailyAverageConsumptionCache);
        $this->productRepository->invalidateDailyAverageConsumptionCache($productIdsToInvalidateDailyAverageConsumptionCache);
        dispatch(new GenerateCacheDailyAverageConsumptionForProductsJob())->onQueue('products');
        dispatch(new CalculateDailyFinancialsJob())->onQueue('financials');
    }

    public function seedDailyFinancials(): void
    {
        // Seed products
        $this->seedProducts();

        // Seed financial line types
        $this->seedFinancialLineTypes();

        // Seed nominal codes
        $this->seedNominalCodes();
    }

    private function seedProducts(): void
    {
        $salesOrderLineQuery = $this->salesOrderLineRepository->withoutDailyFinancialsQuery($this->userTimezone);
        //Log::info('Sale Order Lines needing daily financials seeding: ' . $salesOrderLineQuery->count());

        $salesOrderLineQuery->cursor()->chunk(30000)->each(function ($salesOrderLines) {
            $reportingDailyFinancialCollection = new Collection();
            $invalidatedDatesCollection = new Collection();
            $salesOrderLines->each(function (SalesOrderLine $salesOrderLine) use ($reportingDailyFinancialCollection, $invalidatedDatesCollection, &$i) {
                // Convert the sales order date to the user's timezone and set it to the start of the day
                $convertedDate = Carbon::parse($salesOrderLine->order_date)
                    ->setTimezone($this->userTimezone)
                    ->startOfDay()
                    ->setTimezone('UTC')
                    ->toDateTimeString();

                $reportingDailyFinancialCollection->add(ReportingDailyFinancialDto::from([
                    'reportable_type' => Product::class,
                    'reportable_id' => $salesOrderLine->product_id,
                    'date' => $convertedDate,
                ]));

                $invalidatedDatesCollection->add($convertedDate);
            });

            $this->dailyFinancialRepository->save($reportingDailyFinancialCollection, ReportingDailyFinancial::class);
            $this->dailyFinancialRepository->invalidateForDates($invalidatedDatesCollection->toArray());
        });
    }

    private function seedFinancialLineTypes(): void
    {
        $financialLineTypesQuery = $this->financialLineTypeRepository->unallocatedLinesWithoutDailyFinancialQuery($this->userTimezone);
        //Log::info('Financial line types needing daily financials seeding: ' . $financialLineTypesQuery->count());

        $financialLineTypesQuery->cursor()->chunk(30000)->each(function ($financialLineTypes) {
            $reportingDailyFinancialCollection = new Collection();
            $invalidatedDatesCollection = new Collection();
            $financialLineTypes->each(function ($financialLineType) use ($reportingDailyFinancialCollection, $invalidatedDatesCollection) {
                // Convert the sales order date to the user's timezone and set it to the start of the day
                $convertedDate = Carbon::parse($financialLineType->order_date)
                    ->setTimezone($this->userTimezone)
                    ->startOfDay()
                    ->setTimezone('UTC')
                    ->toDateTimeString();

                $reportingDailyFinancialCollection->add(ReportingDailyFinancialDto::from([
                    'reportable_type' => FinancialLineType::class,
                    'reportable_id' => $financialLineType->id,
                    'date' => $convertedDate,
                ]));

                $invalidatedDatesCollection->add($convertedDate);
            });

            $this->dailyFinancialRepository->save($reportingDailyFinancialCollection, ReportingDailyFinancial::class);
            $this->dailyFinancialRepository->invalidateForDates($invalidatedDatesCollection->toArray());
        });
    }

    private function seedNominalCodes(): void
    {
        $userTimezone = Helpers::getAppTimezone();

        // TODO: Implement once cost invoices is ready
        //        $nominalCodes = NominalCode::query()->whereHas('costInvoiceLines.costInvoice', function ($query) use($userTimezone) {
        //            $query->whereNotExists(function ($subQuery) use ($userTimezone) {
        //                $subQuery->select(DB::raw(1))
        //                    ->from('reporting_daily_financials as rdf')
        //                    ->whereRaw('DATE(CONVERT_TZ(cost_invoices.date, "UTC", "' . $userTimezone '")) = rdf.date')
        //                    ->whereColumn('rdf.reportable_id', '=', 'nominal_code.id')
        //                    ->where('rdf.reportable_type', NominalCode::class);
        //            });
        //        })->whereDoesntHave('costInvoiceLines.costInvoiceAllocations')->get();
        //
        //        $nominalCodes->load('costInvoiceLines.costInvoice');
        //
        //        $records = $nominalCodes->flatMap(function (NominalCode $nominalCode) use ($userTimezone) {
        //            return $nominalCode->costInvoiceLines->map(function (CostInvoiceLine $costInvoiceLine) use ($nominalCode, $userTimezone) {
        //                // Convert the sales order date to the user's timezone and set it to the start of the day
        //                $convertedDate = $costInvoiceLine->costInvoice->date
        //                    ->setTimezone($userTimezone)
        //                    ->startOfDay()
        //                    ->setTimezone('UTC')
        //                    ->toDateTimeString();
        //                return [
        //                    'reportable_type' => NominalCode::class,
        //                    'reportable_id' => $nominalCode->id,
        //                    'date' => $convertedDate,
        //                ];
        //            });
        //        });
        //
        //        $invalidatedDates = array_merge($invalidatedDates, $records->pluck('date')->toArray());
        //        $this->addUniqueRecords($records, $newRecords, $invalidatedDates);
    }
}
