<?php

namespace App\Services\EmailTemplate;

use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Supplier;
use App\Response;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

class EmailTemplatePreviewer
{
    public static function bulkPreview($id, $type, $subject, $body, $v)
    {
        switch ($type) {
            case 'bulk_purchase_order':
                return self::bulkPreivewPO($id, $subject, $body, $v);
            default: throw new Exception('Not supported type.');
        }
    }

    private static function bulkPreivewPO($supplierId, $subject, $body, $v)
    {
        $purchaseOrders = PurchaseOrder::with([])->where('supplier_id', $supplierId)
            ->orderBy('id', 'desc')
            ->select(['id', 'purchase_order_number'])
            ->limit(2)
            ->get();

        if ($purchaseOrders->isEmpty()) {
            return Response::instance()->warning()->addWarning(__('messages.store_email_template.empty_purchase_orders'), Response::CODE_EMPTY_PO, 'id');
        }

        $preview = self::fillTemplate($purchaseOrders->first()['id'], 'PurchaseOrder', $subject, $body, $v);

        if ($v) {
            $preview['{{po_number_list}}'] = $purchaseOrders->implode('purchase_order_number', '<br>');
        } else {
            $preview['subject'] = str_replace('{{po_number_list}}', $purchaseOrders->implode('purchase_order_number', ','), $preview['subject']);
            $preview['body'] = str_replace('{{po_number_list}}', $purchaseOrders->implode('purchase_order_number', '<br>'), $preview['body']);
        }

        return $preview;
    }

    public static function fillTemplate($id, $type, $subject, $body, bool $v = false): array
    {
        $data = self::getModelVariables($id, $type);
        $variables = config('store-email-templates.'.$type);

        preg_match_all("/\{\{([^\}]*)\}\}/", $body, $matches);
        $replaces = $matches[0];
        $extractedVariables = [];
        foreach ($replaces as $replace) {
            if (array_key_exists($replace, $variables)) {
                if ($variables[$replace] == 'sales_order_lines') {
                    $orderLines = self::getSalesOrderLines($id);
                    $body = str_replace($replace, $orderLines, $body);
                    $extractedVariables[$replace] = $orderLines;

                    continue;
                }
                if ($variables[$replace] == 'purchase_order_lines') {
                    $pOrderLines = self::getPurchaseOrderLines($id);
                    $body = str_replace($replace, $pOrderLines, $body);
                    $extractedVariables[$replace] = $pOrderLines;

                    continue;
                }
                if ($variables[$replace] == 'sales_credit_lines') {
                    $scLines = self::getSalesCreditLines($id);
                    $body = str_replace($replace, $scLines, $body);
                    $extractedVariables[$replace] = $scLines;

                    continue;
                }
                if (array_key_exists($variables[$replace], $data)) {
                    $replaced = $data[$variables[$replace]];

                    if (Str::contains($variables[$replace], 'logo')) {
                        $url = $replaced;
                        if (Str::startsWith($url, ['/images', '/storage/images'])) {
                            $url = env('APP_URL').$url;
                        }

                        $replaced = "<img src='$url' height='100px' width='100px'/>";
                    }
                    $body = str_replace($replace, $replaced, $body);
                    $extractedVariables[$replace] = $replaced ?: '';

                    continue;
                }
            }
            $extractedVariables[$replace] = '';
            $body = str_replace($replace, '', $body);
        }
        preg_match_all("/\{\{([^\}]*)\}\}/", $subject, $matches);
        $replaces = $matches[0];

        foreach ($replaces as $replace) {
            if (array_key_exists($replace, $variables) && array_key_exists($variables[$replace], $data)) {
                $subject = str_replace($replace, $data[$variables[$replace]], $subject);
                $extractedVariables[$replace] = $data[$variables[$replace]] ?: '';

                continue;
            }
            $extractedVariables[$replace] = '';
            $subject = str_replace($replace, '', $subject);
        }

        if ($v) {
            return $extractedVariables;
        }

        return ['subject' => $subject, 'body' => $body];
    }

    public static function getModelVariables($id, $type)
    {
        $variables = config('store-email-templates.'.$type);

        /** @var Model $model */
        $model = self::getTemplateModel($type);

        $load = collect($variables)->filter(function ($var) {
            return Str::contains($var, '.');
        })->map(function ($var) {
            return Str::camel(explode('.', $var)[0]);
        })->values()->unique()->filter(function ($var) use ($type) {
            return $type !== 'SalesCredit' || $var !== 'shippingAddress';
        })->toArray();

        $mutators = self::getMutatorMethods($model);

        $mutators = collect($mutators)->map(function ($match) {
            return lcfirst(Str::snake($match));
        });

        $mutators = $mutators->intersect(array_values($variables ?? []))->filter(function ($q) {
            return isset($q) && strlen($q);
        })->toArray();

        $data = $model::with($load)->findOrFail($id);
        if ($type === 'SalesOrder') {
            $data->load('salesOrderLines');
        } elseif ($type === 'PurchaseOrder') {
            $data->load('purchaseOrderLines');
        }
        $data = $data->setAppends($mutators)->toArray();

        return Arr::dot($data);
    }

    public static function getTemplateModel($type)
    {
        switch ($type) {
            case 'SalesOrder':
                return SalesOrder::class;
            case 'PurchaseOrder':
                return PurchaseOrder::class;
            case 'SalesCredit':
                return SalesCredit::class;
            case 'Supplier':
                return Supplier::class;
            default:
                throw new Exception('Model not supported.');
        }
    }

    private static function getSalesOrderLines($id)
    {
        $salesOrderLines = SalesOrderLine::with(['product'])->where('sales_order_id', $id)->get();

        $table = "<table class='et-sales-order-table'>";
        $table .= '<thead><tr><th>SKU</th><th>Product</th><th>QTY</th><th>Price</th><th>Disc%</th><th>Tax%</th><th>Total</th></tr></thead>';
        $table .= '<tbody>';
        foreach ($salesOrderLines as $line) {
            $total = floatval($line->amount) * floatval($line->quantity);
            $sku = $line->product ? $line->product->sku : '';
            $table .= '<tr>';
            $table .= "<td>{$sku}</td>";
            $table .= "<td>{$line->description}</td>";
            $table .= "<td>{$line->quantity}</td>";
            $table .= "<td>{$line->amount}</td>";
            $table .= "<td>{$line->discount}</td>";
            $table .= "<td>{$line->tax_value}</td>";
            $table .= "<td>{$total}</td>";
            $table .= '</tr>';
        }
        $table .= '</tbody>';
        $table .= '</table>';

        return $table;
    }

    private static function getPurchaseOrderLines($id)
    {
        $purchaseOrderLines = PurchaseOrderLine::with(['product'])->where('purchase_order_id', $id)->get();

        $table = "<table class='et-purchase-order-table'>";
        $table .= '<thead><tr><th>SKU</th><th>Product</th><th>QTY</th><th>Price</th><th>Disc%</th><th>Tax%</th><th>Total</th></tr></thead>';
        $table .= '<tbody>';
        foreach ($purchaseOrderLines as $line) {
            $total = floatval($line->amount) * floatval($line->quantity);
            $sku = $line->product ? $line->product->sku : '';
            $table .= '<tr>';
            $table .= "<td>{$sku}</td>";
            $table .= "<td>{$line->description}</td>";
            $table .= "<td>{$line->quantity}</td>";
            $table .= "<td>{$line->amount}</td>";
            $table .= "<td>{$line->discount}</td>";
            $table .= "<td>{$line->tax}</td>";
            $table .= "<td>{$total}</td>";
            $table .= '</tr>';
        }
        $table .= '</tbody>';
        $table .= '</table>';

        return $table;
    }

    private static function getSalesCreditLines($id)
    {
        $salesCreditLines = SalesCreditLine::with(['product'])->where('sales_credit_id', $id)->get();

        $table = "<table class='et-purchase-order-table'>";
        $table .= '<thead><tr><th>SKU</th><th>Product</th><th>QTY</th><th>Price</th><th>Disc%</th><th>Tax%</th><th>Total</th></tr></thead>';
        $table .= '<tbody>';
        foreach ($salesCreditLines as $line) {
            $total = floatval($line->amount) * floatval($line->quantity);
            $sku = $line->product ? $line->product->sku : '';
            $table .= '<tr>';
            $table .= "<td>{$sku}</td>";
            $table .= "<td>{$line->description}</td>";
            $table .= "<td>{$line->quantity}</td>";
            $table .= "<td>{$line->amount}</td>";
            $table .= "<td>{$line->discount}</td>";
            $table .= "<td>{$line->tax}</td>";
            $table .= "<td>{$total}</td>";
            $table .= '</tr>';
        }
        $table .= '</tbody>';
        $table .= '</table>';

        return $table;
    }

    /**
     * Get all of the attribute mutator methods.
     *
     * @param  mixed  $class
     */
    protected static function getMutatorMethods($class): array
    {
        preg_match_all('/(?<=^|;)get([^;]+?)Attribute(;|$)/', implode(';', get_class_methods($class)), $matches);

        return $matches[1];
    }
}
