<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Helpers;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\Setting;
use App\Repositories\NominalCodeRepository;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionLineDataFromSalesOrderFulfillmentLine
{
    private NominalCodeRepository $nominalCodes;

    public function __construct(
        private readonly SalesOrderFulfillmentLine $line,
    ) {
        $this->nominalCodes = app(NominalCodeRepository::class);
    }
    
    /**
     * @throws Throwable
     */
    public function handle(): Collection
    {
        $lineData = collect();
        foreach ($this->getTypeCodeMappings() as $type => $nominalCodeId)
        {
            $lineData->add(AccountingTransactionLineData::from([
                'type' => $type,
                'nominal_code_id' => $nominalCodeId,
                'description' => $this->getDescription(),
                'quantity' => $this->line->quantity,
                'amount' => $this->getAmount(),
                'tax_amount' => 0,
                'tax_rate_id' => null,
                'link_id' => $this->line->id,
                'link_type' => SalesOrderFulfillmentLine::class,
            ]));
        }
        return $lineData;
    }

    private function getTypeCodeMappings(): array
    {
        $salesOrderLine = $this->line->salesOrderLine;
        $salesOrder = $salesOrderLine->salesOrder;
        $warehouse = $this->line->salesOrderFulfillment->warehouse;
        $inventoryAccount = $warehouse->nominal_code_id ?? Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_CONTROL, true);
        return [
            AccountingTransactionLineTypeEnum::DEBIT->value =>
                $this->nominalCodes->getCogsNominalCodeId(
                    $salesOrderLine,
                    $salesOrder->salesChannel->integrationInstance
                ),
            AccountingTransactionLineTypeEnum::CREDIT->value =>
                $inventoryAccount,
        ];
    }

    private function getAmount(): float
    {
        return $this->line->quantity == 0 ?
            0 :
            (
                ($this->line->prorationOfSalesOrderLine * $this->line->salesOrderLine->salesOrderLineFinancial->cogs) /
                $this->line->quantity
            );
    }

    private function getDescription(): string
    {
        return 'Fulfillment for ' . $this->line->salesOrderLine->product->sku;
    }
}