<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Helpers;
use App\Models\SalesCreditReturnLine;
use App\Models\Setting;
use App\Repositories\InventoryMovementRepository;
use App\Repositories\NominalCodeRepository;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionLineDataFromSalesCreditReturnLine
{
    private NominalCodeRepository $nominalCodes;
    private InventoryMovementRepository $movements;

    public function __construct(
        private readonly SalesCreditReturnLine $line,
    ) {
        $this->nominalCodes = app(NominalCodeRepository::class);
        $this->movements = app(InventoryMovementRepository::class);
    }
    
    /**
     * @throws Throwable
     */
    public function handle(): Collection
    {
        $lineData = collect();
        foreach ($this->getTypeCodeMappings() as $type => $nominalCodeId)
        {
            $lineData->add(AccountingTransactionLineData::from([
                'type' => $type,
                'nominal_code_id' => $nominalCodeId,
                'description' => $this->getDescription(),
                'quantity' => $this->line->quantity,
                'amount' => $this->getAmount(),
                'tax_amount' => 0,
                'tax_rate_id' => null,
                'link_id' => $this->line->id,
                'link_type' => SalesCreditReturnLine::class,
            ]));
        }
        return $lineData;
    }

    private function getTypeCodeMappings(): array
    {
        $salesCreditLine = $this->line->salesCreditLine;
        $salesOrder = $salesCreditLine->salesCredit->salesOrder;
        $warehouse = $this->line->salesCreditReturn->warehouse;
        $inventoryAccount = $warehouse->nominal_code_id ?? Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_CONTROL, true);
        return [
            AccountingTransactionLineTypeEnum::DEBIT->value =>
                $inventoryAccount,
            AccountingTransactionLineTypeEnum::CREDIT->value =>
                $this->nominalCodes->getCogsNominalCodeId(
                $salesCreditLine,
                $salesOrder->salesChannel->integrationInstance
            ),
        ];
    }

    private function getAmount(): float
    {
        return $this->line->quantity == 0 ?
            0 :
            (
                $this->movements->getTotalAmount($this->line->inventoryMovements) /
                $this->line->quantity
            );
    }

    private function getDescription(): string
    {
        $salesCreditLine = $this->line->salesCreditLine;
        return 'Return ' . ($salesCreditLine->product?->sku ? $salesCreditLine->product->sku . ': ' : '') . $salesCreditLine->description;
    }
}