<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Helpers;
use App\Models\NominalCode;
use App\Models\PurchaseOrderLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use Throwable;

class BuildAccountingTransactionLineDataFromPurchaseOrderLine
{
    public function __construct(
        private readonly PurchaseOrderLine $line,
    ) {}
    
    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionLineData
    {
        return AccountingTransactionLineData::from([
            'type' => AccountingTransactionLineTypeEnum::PURCHASE_ORDER_LINE,
            'nominal_code_id' => $this->getNominalCodeId(),
            'description' => $this->getDescription(),
            'quantity' => $this->line->quantity,
            'amount' => $this->getAmount(),
            'tax_amount' => $this->line->tax_allocation_in_tenant_currency,
            'tax_rate_id' => $this->line->tax_rate_id,
            'link_id' => $this->line->id,
            'link_type' => PurchaseOrderLine::class,
        ]);
    }

    private function getDescription(): string
    {
        return ($this->line->product?->sku ? $this->line->product->sku . ': ' : '') . $this->line->description;
    }

    private function getAmount(): float
    {
        return $this->line->quantity == 0 ?
            0 :
            (
                $this->line->purchaseOrder->is_tax_included ?
                    ($this->line->subtotal_in_tenant_currency - $this->line->tax_allocation_in_tenant_currency) / $this->line->quantity :
                    $this->line->subtotal_in_tenant_currency / $this->line->quantity
            );
    }

    private function getNominalCodeId(): int
    {
        $purchaseOrder = $this->line->purchaseOrder;
        if ($purchaseOrder->dropshipping || !$purchaseOrder->destinationWarehouse ) {
            return Helpers::setting(Setting::KEY_NC_MAPPING_COGS);
        }

        return Helpers::setting(Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES);
    }
}