<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Models\Currency;
use App\Models\WarehouseTransferShipment;
use App\Models\WarehouseTransferShipmentLine;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionDataFromWarehouseTransferShipment
{
    public function __construct(
        private readonly WarehouseTransferShipment|FinancialDocumentInterface $warehouseTransferShipment,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        $warehouseTransfer = $this->warehouseTransferShipment->warehouseTransfer;
        $linesCollection = $this->getLines();
        $name = $warehouseTransfer->fromWarehouse->name . '->' . $warehouseTransfer->toWarehouse->name;
        $reference = $name . ': ' . $warehouseTransfer->warehouse_transfer_number . ': Shipment ' . $this->warehouseTransferShipment->id;
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::WAREHOUSE_TRANSFER_SHIPMENT,
            'link_id' => $this->warehouseTransferShipment->id,
            'link_type' => WarehouseTransferShipment::class,
            'name' => $name,
            'email' => null,
            'transaction_date' => $this->warehouseTransferShipment->shipped_at,
            'reference' => $reference,
            'is_tax_included' => false,
            'currency_code' => Currency::default()->code,
            'currency_rate' => 1,
            'is_batchable' => false,
            'parent_id' => $this->warehouseTransferShipment->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->where('type', AccountingTransactionLineTypeEnum::DEBIT)
                ->sum(fn ($line) => abs($line->quantity * $line->amount)),
        ]);
    }

    private function getLines(): Collection
    {
        return $this->warehouseTransferShipment->shipmentLines->map(function (WarehouseTransferShipmentLine $line)
        {
            return (new BuildAccountingTransactionLineDataFromWarehouseTransferShipmentLine($line))->handle();
        })->flatten();
    }
}