<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionDataFromStockTake
{
    public function __construct(
        private readonly StockTake|FinancialDocumentInterface $stockTake,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        $stockTake = $this->stockTake;

        $linesCollection = $this->getLines();
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::STOCK_TAKE,
            'link_id' => $stockTake->id,
            'link_type' => StockTake::class,
            'name' => null,
            'email' => null,
            'transaction_date' => $stockTake->date_count,
            'reference' => "Stock Take {$stockTake->id} for {$stockTake->warehouse->name}",
            'is_tax_included' => false,
            'currency_code' => null,
            'currency_rate' => null,
            'is_batchable' => false,
            'parent_id' => $stockTake->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->where('type', AccountingTransactionLineTypeEnum::DEBIT)
                ->sum(fn ($line) => abs($line->quantity * $line->amount)),
        ]);
    }

    private function getLines(): Collection
    {
        return $this->stockTake->stockTakeItems->where('quantity_adjusted', '!=', 0)->map(function (StockTakeItem $line)
        {
            return (new BuildAccountingTransactionLineDataFromStockTakeItem($line))->handle();
        })->flatten();
    }
}