<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionTypeEnum;
use App\Enums\FinancialLineClassificationEnum;
use App\Models\FinancialLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Throwable;

/*
 * TODO: Consideration: If not fulfilled, should create additional account transaction (journal)
 *  to Dr Line Nominal Code Cr Prepayments
 */
class BuildAccountingTransactionDataFromSalesOrder
{
    public function __construct(
        private readonly SalesOrder|FinancialDocumentInterface $salesOrder,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        $salesOrder = $this->salesOrder;

        // Validate financial cache before doing calculations that rely on it
        $manager = new SalesOrderLineFinancialManager(Arr::wrap($salesOrder->id));
        $manager->calculate();

        $linesCollection = $this->getLines();
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::SALES_ORDER_INVOICE,
            'link_id' => $salesOrder->id,
            'link_type' => SalesOrder::class,
            'name' => $salesOrder->customer?->name,
            'email' => $salesOrder->customer?->email,
            'transaction_date' => $salesOrder->order_date,
            'reference' => $salesOrder->sales_order_number,
            'is_tax_included' => $salesOrder->is_tax_included,
//            'currency_code' => $salesOrder->currency->code,
//            'currency_rate' => $salesOrder->currency_rate,
            'currency_code' => $salesOrder->currencyTenantSnapshot->code,
            'currency_rate' => 1,
            'is_batchable' => $this->areSalesOrderInvoicesBatchableForSalesChannel(),
            'parent_id' => $salesOrder->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->sum(fn ($line) => $line->quantity * $line->amount),
            'is_sync_enabled' => $salesOrder->salesChannel->integrationInstance->integration_settings['sync_sales_order_invoices_to_accounting'] ?? false
        ]);
    }

    private function getLines(): Collection
    {
        $linesCollection = $this->getAccountingTransactionLinesFromSalesOrderLines();
        return $linesCollection->merge($this->getAccountingTransactionLinesFromFinancialLines());
    }

    private function areSalesOrderInvoicesBatchableForSalesChannel(): bool
    {
        return $this->salesOrder->salesChannel->integrationInstance->integration_settings['batch_sales_order_invoices'] ?? false;
    }

    private function getAccountingTransactionLinesFromSalesOrderLines(): Collection
    {
        return $this->salesOrder->salesOrderLines->map(function (SalesOrderLine $line)
        {
            return (new BuildAccountingTransactionLineDataFromSalesOrderLine($line))->handle();
        });
    }

    private function getAccountingTransactionLinesFromFinancialLines(): Collection
    {
        return $this->salesOrder->financialLines
            ->filter(fn (FinancialLine $line) => $line->financialLineType->classification === FinancialLineClassificationEnum::REVENUE)
            ->map(function (FinancialLine $line) {
                return (new BuildAccountingTransactionLineDataFromFinancialLine($line))->handle();
            });
    }
}