<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionDataFromSalesCreditReturn
{
    public function __construct(
        private readonly SalesCreditReturn|FinancialDocumentInterface $salesCreditReturn,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        $salesCreditReturn = $this->salesCreditReturn;

        // Validate financial cache before doing calculations that rely on it
        if ($salesCreditReturn->salesCredit->salesOrder) {
            $manager = new SalesOrderLineFinancialManager(Arr::wrap($salesCreditReturn->salesCredit->salesOrder->id));
            $manager->calculate();
        }

        $linesCollection = $this->getLines();
        $name = $this->salesCreditReturn->salesCredit->customer?->name;
        $reference = $this->getReference();
        if ($name) {
            $reference = $name . ': ' . $reference;
        }
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::CUSTOMER_RETURN,
            'link_id' => $salesCreditReturn->id,
            'link_type' => SalesCreditReturn::class,
            'name' => $name,
            'email' => $this->salesCreditReturn->salesCredit->customer?->email,
            'transaction_date' => $salesCreditReturn->received_at,
            'reference' => $reference,
            'is_tax_included' => false,
            'currency_code' => null,
            'currency_rate' => null,
            'is_batchable' => false,
            'parent_id' => $salesCreditReturn->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->where('type', AccountingTransactionLineTypeEnum::DEBIT)
                ->sum(fn ($line) => abs($line->quantity * $line->amount)),
        ]);
    }

    private function getLines(): Collection
    {
        return $this->salesCreditReturn->salesCreditReturnLines->map(/**
         * @throws Throwable
         */ function (SalesCreditReturnLine $line)
        {
            return (new BuildAccountingTransactionLineDataFromSalesCreditReturnLine($line))->handle();
        })->flatten();
    }

    private function getReference(): string
    {
        return "Customer Return #{$this->salesCreditReturn->id} through {$this->salesCreditReturn->salesCredit->sales_credit_number} ({$this->salesCreditReturn->salesCredit->salesOrder->sales_order_number})";
    }
}