<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionTypeEnum;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionDataFromSalesCredit
{
    public function __construct(
        private readonly SalesCredit|FinancialDocumentInterface $salesCredit,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        $salesCredit = $this->salesCredit;

        // Validate financial cache before doing calculations that rely on it
        if ($salesCredit->salesOrder) {
            $manager = new SalesOrderLineFinancialManager(Arr::wrap($salesCredit->salesOrder->id));
            $manager->calculate();
        }

        $linesCollection = $this->getLines();
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::SALES_CREDIT,
            'link_id' => $salesCredit->id,
            'link_type' => SalesCredit::class,
            'name' => $salesCredit->customer?->name,
            'email' => $salesCredit->customer?->email,
            'transaction_date' => $salesCredit->credit_date,
            'reference' => $salesCredit->sales_credit_number,
            'is_tax_included' => $salesCredit->is_tax_included ?? false,
//            'currency_code' => $salesCredit->currency->code,
//            'currency_rate' => $salesCredit->currency_rate,
            'currency_code' => $salesCredit->currencyTenantSnapshot->code,
            'currency_rate' => 1,
            'is_batchable' => false,
            'parent_id' => $salesCredit->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->sum(fn ($line) => $line->quantity * $line->amount),
        ]);
    }

    private function getLines(): Collection
    {
        return $this->salesCredit->salesCreditLines->map(function (SalesCreditLine $line)
        {
            return (new BuildAccountingTransactionLineDataFromSalesCreditLine($line))->handle();
        });
    }
}