<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionDataFromPurchaseOrder
{
    public function __construct(
        private readonly PurchaseOrder|FinancialDocumentInterface $purchaseOrder,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        $purchaseOrder = $this->purchaseOrder;

        $linesCollection = $this->getLines();
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::PURCHASE_ORDER,
            'link_id' => $purchaseOrder->id,
            'link_type' => PurchaseOrder::class,
            'name' => $purchaseOrder->supplier->name,
            'email' => $purchaseOrder->supplier->email,
            'transaction_date' => $purchaseOrder->purchase_order_date,
            'reference' => $purchaseOrder->purchase_order_number,
            'is_tax_included' => $purchaseOrder->is_tax_included ?? false,
            //'currency_code' => $purchaseOrder->currency->code,
            //'currency_rate' => $purchaseOrder->currency_rate,
            'currency_code' => $purchaseOrder->currencyTenantSnapshot->code,
            'currency_rate' => 1,
            'is_batchable' => false,
            'parent_id' => $purchaseOrder->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->sum(fn ($line) => $line->quantity * $line->amount),
        ]);
    }

    private function getLines(): Collection
    {
        return $this->purchaseOrder->purchaseOrderLines->map(function (PurchaseOrderLine $line)
        {
            return (new BuildAccountingTransactionLineDataFromPurchaseOrderLine($line))->handle();
        });
    }
}