<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Models\InventoryAdjustment;
use App\Models\PurchaseOrderLine;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionDataFromInventoryAdjustment
{
    public function __construct(
        private readonly InventoryAdjustment|FinancialDocumentInterface $inventoryAdjustment,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        // TODO: Old code had filtered out adjustments with quantity 0
        $inventoryAdjustment = $this->inventoryAdjustment;

        $linesCollection = $this->getLines();
        $name = $inventoryAdjustment->link instanceof PurchaseOrderLine ?
            $inventoryAdjustment->link->purchaseOrder->supplier->name :
            null;
        $reference = $this->getReference();
        if ($name) {
            $reference = $name . ': ' . $reference;
        }
        if ($this->inventoryAdjustment->link instanceof PurchaseOrderLine) {
            $reference .= ' for ' . $this->inventoryAdjustment->link->purchaseOrder->purchase_order_number;
        }
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::INVENTORY_ADJUSTMENT,
            'link_id' => $inventoryAdjustment->id,
            'link_type' => InventoryAdjustment::class,
            'name' => $name,
            'email' => $inventoryAdjustment->link instanceof PurchaseOrderLine ?
                $inventoryAdjustment->link->purchaseOrder->supplier->email :
                null,
            'transaction_date' => $inventoryAdjustment->adjustment_date,
            'reference' => $reference,
            'is_tax_included' => false,
            'currency_code' => null,
            'currency_rate' => null,
            'is_batchable' => false,
            'parent_id' => $inventoryAdjustment->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->where('type', AccountingTransactionLineTypeEnum::DEBIT)
                ->sum(fn ($line) => abs($line->quantity * $line->amount)),
        ]);
    }

    private function getLines(): Collection
    {
        return (new BuildAccountingTransactionLineDataFromInventoryAdjustment($this->inventoryAdjustment))->handle();
    }

    private function getReference(): string
    {
        return "Stock Adjustment {$this->inventoryAdjustment->product->sku} (ID: {$this->inventoryAdjustment->id})";
    }
}