<?php
/**
 * Created by Inspiring IT Solutions.
 * User: a.elsaidy<ahmad.elsaidy@hotmail.com>
 * Date: 4/3/2019
 * Time: 3:48 PM
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at.
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace App\SDKs\WalmartSDK;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use phpseclib\Crypt\Random;

class WalmartAPI
{
    const API_URL = 'https://marketplace.walmartapis.com/';

    const API_VERSION = 'v3';

    /**
     * Marker places.
     */
    const MARKET_PLACE_US = 'US';

    const MARKET_PLACE_CA = 'CA';

    private $marketPlaceURI;

    private $markerPlace = self::MARKET_PLACE_US;

    private $isOAuth = true;

    /**
     *  Digital Signature authentication arguments.
     */
    private $consumerId;

    private $privateKey;

    private $wmConsumerChannelType;

    /**
     * OAuth authentication arguments.
     */
    private $clientID;

    private $clientSecret;

    private $accessToken;

    /**
     * WalmartAPI constructor.
     *
     *
     * @throws Exception
     */
    public function __construct(
        array $credentials,
        ?string $marketPlace = null,
        bool $isOAuth = true
    ) {
        if ($isOAuth) {
            /**
             * Check and set credentials if OAuth.
             */
            if (! isset($credentials['clientID'])
           && ! isset($credentials['clientSecret'])
            ) {
                throw new \InvalidArgumentException('clientID and clientSecret are required when OAuth Authentication.');
            }

            $this->clientID = $credentials['clientID'];
            $this->clientSecret = $credentials['clientSecret'];
            if (isset($credentials['accessToken'])) {
                $this->accessToken = $credentials['accessToken'];
            }
        } else {
            /**
             * Check and set credentials if non-OAuth.
             */
            if (! isset($credentials['consumerId'])
           && ! isset($credentials['privateKey'])
           && ! isset($credentials['wmConsumerChannelType'])
            ) {
                throw new \InvalidArgumentException('consumerId and privateKey are required when Digital Signature Authentication.');
            }

            $this->consumerId = $credentials['consumerId'];
            $this->privateKey = $credentials['privateKey'];
            $this->wmConsumerChannelType = $credentials['wmConsumerChannelType'];
            $this->isOAuth = false;
        }

        // marketPlaceURI is US
        $this->marketPlaceURI = '';
        // Check MarkerPlaceUri is CA
        if (! empty($marketPlace) && $marketPlace == self::MARKET_PLACE_CA) {
            $this->marketPlaceURI = '/ca';
            $this->markerPlace = self::MARKET_PLACE_CA;
        }
    }

    /**
     * Generate OAuth url.
     */
    public static function getOAuthURL(array $params): string
    {
        if (! array_key_exists('state', $params)) {
            throw new \InvalidArgumentException('state parameter required');
        }

        if (! array_key_exists('scope', $params)) {
            throw new \InvalidArgumentException('scope parameter required');
        }

        if (! array_key_exists('clientId', $params)) {
            throw new \InvalidArgumentException('clientId parameter required');
        }

        if (! array_key_exists('redirectUri', $params)) {
            throw new \InvalidArgumentException('redirectUri parameter required');
        }

        if (! array_key_exists('nonce', $params)) {
            throw new \InvalidArgumentException('nonce parameter required');
        }

        $url = 'https://retaillink.login.wal-mart.com/authorize?'; // I think static
        $urlParams = [
            'clientId' => $params['clientId'],
            'redirectUri' => $params['redirectUri'],
            'clientType' => 'seller',
            'responseType' => 'code',
            'state' => $params['state'],
            'nonce' => $params['nonce'],
            'scope' => implode($params['scope'], ' '),
        ];

        return $url.http_build_query($urlParams, null, '&', PHP_QUERY_RFC3986);
    }

    /**
     * Get Client Access token (OAuth).
     *
     * @return array|bool
     */
    public function getAccessToken()
    {
        if (! $this->isOAuth) {
            return ['error' => 'getAccessToken function only for OAuth authentication only.'];
        }

        $params = ['grant_type' => 'client_credentials'];

        return $this->makeRequest('token', $params, 'POST');
    }

    /**
     * get All Orders.
     *
     *
     * @return array|bool
     */
    public function getOrders(array $options = [])
    {
        $endpoint = 'orders';

        if (empty($options)) {
            $options = [
                'body' => [],
                'params' => [
                    'createdStartDate' => date('Y-m-d', time() - 5184000),
                    'limit' => 10,
                ],
            ];
        } else {
            if (empty($options['nextCursor'])) {
                $params = $options;
            } else {
                /**
                 * Covert Next Cursor to array of options.
                 */
                $nextCursor = str_replace('?', '', $options['nextCursor']);
                parse_str($nextCursor, $params);

                /**
                 * Fix Date format for "createdStartDate" and "createdEndDate".
                 */
                if (isset($params['createdStartDate'])) {
                    $params['createdStartDate'] = date('Y-m-d', strtotime($params['createdStartDate']));
                }

                if (isset($params['createdEndDate'])) {
                    $params['createdEndDate'] = date('Y-m-d', strtotime($params['createdEndDate']));
                }
            }

            $options = [
                'body' => [],
                'params' => $params,
            ];
        }

        $response = $this->makeRequest($endpoint, $options);

        /**
         * If response is null then return empty object as Walmart object.
         */
        if (is_null($response)) {
            return [
                'list' => [
                    'meta' => [
                        'totalCount' => 0,
                        'nextCursor' => null,
                    ],
                    'elements' => [
                        'order' => [],
                    ],
                ],
            ];
        }

        return $response;
    }

    /**
     * Acknowledge Order.
     *
     *
     * @return array|bool
     */
    public function acknowledgeOrder(string $purchaseOrderId)
    {
        $uri = 'orders/'.$purchaseOrderId.'/acknowledge';

        return $this->makeRequest($uri, [], 'POST');
    }

    /**
     * Mark as Shipped and add tracking number.
     *
     *
     *
     * @return array|bool
     *
     * @throws Exception
     */
    public function shipOrder(
        string $purchaseOrderId,
        string $carrier,
        string $trackingNumber,
        ?string $shippingDate = null
    ) {
        $uri = 'orders/'.$purchaseOrderId.'/shipping';
        if (is_null($shippingDate)) {
            $shippingDate = time();
        }

        $params = [
            'orderShipment' => [
                'orderLines' => [
                    [
                        'lineNumber' => 1,
                        'orderLineStatuses' => [
                            [
                                'status' => 'Shipped',
                                'statusQuantity' => [
                                    'unitOfMeasurement' => 'Each',
                                    'amount' => 1,
                                ],
                                'trackingInfo' => [
                                    'shipDateTime' => $shippingDate,
                                    'carrierName' => [
                                        'carrier' => $carrier,
                                    ],
                                    'methodCode' => 'Standard',
                                    'trackingNumber' => $trackingNumber,
                                    'trackingURL' => 'https://www.17track.net/',
                                ],
                            ],
                        ],
                    ],
                ],
            ],
        ];

        return $this->makeRequest($uri, $params, 'POST');
    }

    /**
     * Make GuzzleHttp Request.
     *
     *
     * @return array|bool
     */
    public function makeRequest($uri, $params, ?string $method = null)
    {
        if (empty($method)) {
            $method = 'GET';
        }

        $contentType = 'json';
        $client = new Client();
        $url = $this->buildURL($uri, $params, $method);
        dump($url);

        try {
            $options = $this->setOptions($url, $method, $params, $contentType);
        } catch (Exception $exception) {
            return ['error' => 'privateKey set null or invalid'];
        }

        try {
            $response = $client->request($method, $url, $options);

            return $this->formatResponse(
                $response->getBody()->getContents(),
                $contentType
            );
        } catch (GuzzleException $exception) {
            if ($exception->hasResponse()) {
                $response = $exception->getResponse()->getBody()->getContents();
                $response = $this->formatResponse($response, $contentType);

                $error = $response['errors']['error'][0] ?? $response['error'][0];

                $walmart_code = $error['code'] ?? null;
                $walmart_code = explode('.', $walmart_code)[0] ?? null;

                if ($walmart_code == Enum::ERROR_CONTENT_NOT_FOUND) {
                    return null;
                }

                if (isset($error['description'])) {
                    return ['error' => $error['description']];
                }
            }

            if ($exception->getCode() == 404) {
                return ['error' => 'Check marketplace, Try change it.'];
            }

            return ['error' => $exception->getMessage()];
        }
    }

    /**
     * Build URL Request.
     */
    public function buildURL(string $uri, $params, string $method): string
    {
        $requestURL = self::API_URL.
                  self::API_VERSION.
                  $this->marketPlaceURI.
                  '/'.$uri;

        if ($method == 'GET') {
            $queryString = http_build_query($params['params']);

            $requestURL .= empty($queryString) ? '' : ('?'.$queryString);
        } else {
            if (! is_array($params['body'])) {
                if (strlen($params['body']) < 100) {
                    $requestURL .= http_build_query($params['body']);
                }
            }
        }

        return $requestURL;
    }

    /**
     * Decode url back to normal to nextCursor issue. automatic url encoding.
     */
    public function formatURL($requestUrl): string
    {
        //decode url back to normal to nextCursor issue. automatic url encoding
        $requestUrl = rawurldecode($requestUrl);

        return $requestUrl;
    }

    /**
     * Define request API headers.
     *
     *
     * @throws Exception
     */
    private function setOptions(
        $requestUrl,
        $requestMethod,
        array $body = [],
        ?string $contentType = null
    ): array {
        if (empty($contentType)) {
            $contentType = 'json';
        }

        $options = [
            'headers' => [
                'Host' => 'marketplace.walmartapis.com',
                'Content-Type' => 'application/'.$contentType,
                'Accept' => 'application/'.$contentType,
                'WM_SVC.NAME' => 'Walmart Marketplace',
                'WM_QOS.CORRELATION_ID' => base64_encode(Random::string(16)),
            ],
        ];
        if ($this->isOAuth) {
            $options['headers']['Authorization'] = base64_encode($this->clientID.':'
                                                            .$this->clientSecret);
            if (! empty($this->accessToken)) {
                $options['headers']['WM_SEC.ACCESS_TOKEN'] = $this->accessToken;
            }
        } else {
            $timestamp = Utils::getMilliseconds();
            $signature = Signature::calculateSignature(
                $this->consumerId,
                $this->privateKey,
                $requestUrl,
                $requestMethod,
                $timestamp
            );

            if ($this->markerPlace == self::MARKET_PLACE_CA) {
                $options['headers']['WM_TENANT_ID'] = 'WALMART.CA';
                $options['headers']['WM_LOCALE_ID'] = 'en_CA';
            }
            $options['headers']['WM_SEC.TIMESTAMP'] = $timestamp;
            $options['headers']['WM_SEC.AUTH_SIGNATURE'] = $signature;
            $options['headers']['WM_CONSUMER.ID'] = $this->consumerId;
            $options['headers']['WM_CONSUMER.CHANNEL.TYPE'] = $this->wmConsumerChannelType;
        }

        if ($requestMethod == 'POST') {
            if ($contentType == 'xml') {
                $options['body'] = $body;
            } elseif ($contentType == 'json') {
                $options['form_params'] = $body;
                //            $options["json"] = $body;
            }
        }

        return $options;
    }

    /**
     * Parsing XML with namespaces doesn't seem to work for Guzzle,.
     */
    public function stripXmlNamespaces($xml): string
    {
        /**
         * Parsing XML with namespaces doesn't seem to work for Guzzle,
         * so using regex to remove them.
         */
        $xml = Utils::stripNamespacesFromXml($xml);

        /**
         * Intercept response and replace body with cleaned up XML.
         */

        return $xml;
    }

    /**
     * Format XML/Json Response.
     */
    public function formatResponse($data, ?string $type = null): array
    {
        if (empty($type)) {
            $type = 'json';
        }

        if ($type == 'xml') {
            $xml = $this->stripXmlNamespaces($data);

            return XMLFormatter::convert($xml);
        } else {
            return json_decode($data, true);
        }
    }
}
