<?php
/**
 * Created by Inspiring IT Solutions.
 * User: a.elsaidy<ahmad.elsaidy@hotmail.com>
 * Date: 4/3/2019
 * Time: 10:10 PM
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at.
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace App\SDKs\WalmartSDK;

use phpseclib\Crypt\RSA;

class Signature
{
    /**
     * @var string Consumer ID provided by Developer Portal
     */
    public $consumerId;

    /**
     * @var string Base64 Encoded Private Key provided by Developer Portal
     */
    public $privateKey;

    /**
     * @var string URL of API request being made
     */
    public $requestUrl;

    /**
     * @var string HTTP request method for API call (GET/POST/PUT/DELETE/OPTIONS/PATCH)
     */
    public $requestMethod;

    /**
     * You may optionally instantiate as an object. This is useful for repeated calls to getSignature();.
     */
    public function __construct(string $consumerId, string $privateKey, string $requestUrl, string $requestMethod)
    {
        $this->consumerId = $consumerId;
        $this->privateKey = $privateKey;
        $this->requestUrl = $requestUrl;
        $this->requestMethod = $requestMethod;
    }

    /**
     * Get signature with optional timestamp. If using Signature class as object, you can repeatedly call this
     * method to get a new signature without having to provide $consumerId, $privateKey, $requestUrl, $requestMethod
     * every time.
     *
     *
     * @throws \Exception
     */
    public function getSignature(?string $timestamp = null): string
    {
        if (is_null($timestamp) || ! is_numeric($timestamp)) {
            $timestamp = self::getMilliseconds();
        }

        return self::calculateSignature(
            $this->consumerId,
            $this->privateKey,
            $this->requestUrl,
            $this->requestMethod,
            $timestamp
        );
    }

    /**
     * Static method for quick calls to calculate a signature.
     *
     * @link https://developer.walmartapis.com/#authentication
     *
     * @throws \Exception
     */
    public static function calculateSignature(string $consumerId, string $privateKey, string $requestUrl, string $requestMethod, ?string $timestamp = null): string
    {
        if (is_null($timestamp) || ! is_numeric($timestamp)) {
            $timestamp = self::getMilliseconds();
        }

        /**
         * Append values into string for signing.
         */
        $message = $consumerId."\n".$requestUrl."\n".strtoupper($requestMethod)."\n".$timestamp."\n";

        /**
         * Get RSA object for signing.
         */
        $rsa = new RSA();
        $decodedPrivateKey = base64_decode($privateKey);
        $rsa->setPrivateKeyFormat(RSA::PRIVATE_FORMAT_PKCS8);
        $rsa->setPublicKeyFormat(RSA::PRIVATE_FORMAT_PKCS8);

        /**
         * Load private key.
         */
        if ($rsa->loadKey($decodedPrivateKey, RSA::PRIVATE_FORMAT_PKCS8)) {
            /**
             * Make sure we use SHA256 for signing.
             */
            $rsa->setHash('sha256');
            $rsa->setSignatureMode(RSA::SIGNATURE_PKCS1);
            $signed = $rsa->sign($message);

            /**
             * Return Base64 Encode generated signature.
             */
            return base64_encode($signed);
        } else {
            throw new \Exception('Unable to load private key', 1446780146);
        }
    }

    /**
     * Get current timestamp in milliseconds.
     */
    public static function getMilliseconds(): float
    {
        return round(microtime(true) * 1000);
    }
}
