<?php

namespace App\SDKs\ShipStation\Services;

use App\SDKs\ShipStation\HasAttributes;
use App\SDKs\ShipStation\Model\Order;
use App\SDKs\ShipStation\Requests\ListOrdersRequest;
use App\SDKs\ShipStation\Responses\BulkCreateUpdateOrdersResponse;
use App\SDKs\ShipStation\Responses\CreateUpdateOrdersResponse;
use App\SDKs\ShipStation\Responses\ListOrdersResponse;
use App\SDKs\ShipStation\ShipStation;
use App\SDKs\ShipStation\ShipStationException;
use Symfony\Component\HttpFoundation\Request;

class Orders extends ShipStation
{
    /**
     * Get a List of ShipStation orders.
     *
     *
     * @throws ShipStationException
     */
    public function getList(?ListOrdersRequest $listOrdersRequest = null): ListOrdersResponse
    {
        $query = $listOrdersRequest ? $listOrdersRequest->getQueryParams() : null;

        return new ListOrdersResponse($this->request($query ? "?{$query}" : ''));
    }

    /**
     * Get a single ShipStation order by ShipStation's order id.
     */
    public function view(string $orderId): array
    {
        return $this->request($orderId);
    }

    /**
     * Delete order from ShipStation.
     * Removes order from ShipStation's UI.
     * Note this is a "soft" delete action so the order will still exist in the database,
     * but will be set to inactive.
     */
    public function delete(string $orderId): array
    {
        return $this->request($orderId, [], Request::METHOD_DELETE);
    }

    /**
     * Create a ShipStation order or update it If the <b>orderKey</b> is specified and found,
     * This call does not currently support partial updates;
     * the entire resource must be provided in the body of the request.
     */
    public function create(Order $order): CreateUpdateOrdersResponse
    {
        return new CreateUpdateOrdersResponse($this->request('/createorder', $order->toJson(), Request::METHOD_POST));
    }

    /**
     * Update a ShipStation order.
     */
    public function update(string $orderKey, Order $order): array
    {
        $order->orderKey = $orderKey;

        return $this->request('/createorder', $order->toJson(), Request::METHOD_POST);
    }

    /**
     * This endpoint can be used to create or update multiple orders in one request.
     * If the <b/>orderKey</b> is specified in an order, the existing order with that key will be updated.
     * This call does not currently support partial updates; the entire resource must be provided in the body of the request.
     *
     * @param  Order[]  $orders
     *
     * @throws ShipStationException
     */
    public function bulkCreateUpdate(array $orders): BulkCreateUpdateOrdersResponse
    {
        $orders = json_encode(HasAttributes::attributesToArray($orders));

        return new BulkCreateUpdateOrdersResponse($this->request('/createorders', $orders, Request::METHOD_POST));
    }
}
