<?php

namespace App\SDKs\ShipStation;

use Illuminate\Support\Str;

trait HasAttributes
{
    protected $casts = [];

    protected $attributes = [];

    public function __construct(array $attributes = [])
    {
        $this->setAttributes($attributes);
    }

    /**
     * Set attributes as properties.
     */
    public function setAttributes(array $attributes)
    {
        foreach ($attributes as $attribute => $value) {
            $this->setAttribute($attribute, $value);
        }
    }

    protected function setAttribute($name, $value)
    {
        // check if it has a mutator
        $mutatorFunctionName = 'set'.Str::studly($name).'Attribute';
        if (method_exists($this, $mutatorFunctionName)) {
            return $this->{$mutatorFunctionName}($value);
        }

        $this->attributes[$name] = $value;
    }

    protected function getAttribute($name)
    {
        if (! $name || ! array_key_exists($name, $this->attributes)) {
            return null;
        }

        // check if it has a mutator
        $mutatorFunctionName = 'get'.Str::studly($name).'Attribute';
        if (method_exists($this, $mutatorFunctionName)) {
            return $this->{$mutatorFunctionName}();
        }

        if (array_key_exists($name, $this->casts)) {
            return new $this->casts[$name]($this->attributes[$name]);
        }

        return $this->attributes[$name];
    }

    public function __set($name, $value)
    {
        $this->setAttribute($name, $value);
    }

    public function __get($name)
    {
        return $this->getAttribute($name);
    }

    /**
     * {@inheritDoc}
     */
    public function toArray()
    {
        return static::attributesToArray($this->attributes);
    }

    /**
     * Convert array of attribute(contains objects and arrays) to associated array.
     */
    public static function attributesToArray(array $attributes = []): array
    {
        foreach ($attributes as $key => $value) {
            if ($value instanceof Base) {
                $attributes[$key] = $value->toArray();
            } elseif (is_array($value)) { // recursion if array.
                $attributes[$key] = static::attributesToArray($value);
            } else {
                $attributes[$key] = $value;
            }
        }

        return $attributes;
    }
}
