<?php

namespace App\SDKs\Magento;

use Closure;
use Exception;
use InvalidArgumentException;
use OAuth\Common\Http\Client\AbstractClient;
use OAuth\Common\Http\Uri\UriInterface;

class HttpClient extends AbstractClient
{
    public function __construct(protected ?Closure $afterRetrievingResponse = null, $userAgent = 'SKU.io')
    {
        parent::__construct($userAgent);
    }

    /**
     * Any implementing HTTP providers should send a request to the provided endpoint with the parameters.
     * They should return, in string form, the response body and throw an exception on error.
     *
     * @param  mixed  $requestBody
     *
     * @throws Exception
     */
    public function retrieveResponse(UriInterface $endpoint, $requestBody, array $extraHeaders = [], string $method = 'POST'): array
    {
        // Normalize method name
        $method = strtoupper($method);

        $this->normalizeHeaders($extraHeaders);

        if ($method === 'GET' && ! empty($requestBody)) {
            throw new InvalidArgumentException('No body expected for "GET" request.');
        }

        /**
         * Content-Type: application/json if POST request.
         */
        if (($method === 'POST' || $method === 'PUT') && is_array($requestBody)) {
            $requestBody = json_encode($requestBody);

            $extraHeaders['Content-Type'] = 'Content-Type: application/json';
            $extraHeaders['Content-length'] = 'Content-length: '.strlen($requestBody);
        } else {
            $extraHeaders['Content-Type'] = 'Content-Type: application/x-www-form-urlencoded';
        }

        $host = 'Host: '.$endpoint->getHost();
        // Append port to Host if it has been specified
        if ($endpoint->hasExplicitPortSpecified()) {
            $host .= ':'.$endpoint->getPort();
        }

        $extraHeaders['Host'] = $host;
        $extraHeaders['Connection'] = 'Connection: close';
        $extraHeaders['User-Agent'] = $this->userAgent;

        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_URL => $endpoint->getAbsoluteUri(),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 60,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_POSTFIELDS => $requestBody,
            CURLOPT_HTTPHEADER => array_values($extraHeaders),
            CURLOPT_USERAGENT => $this->userAgent,
        ]);

        $response = curl_exec($curl);
        $err = curl_error($curl);
        $statusCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        curl_close($curl);

        if ($this->afterRetrievingResponse) {
            call_user_func_array($this->afterRetrievingResponse, [$response, $statusCode ?: $err]);
        }

        if ($err) {
            throw new MagentoException('cURL Error #:'.$err, $statusCode);
        }

        if ($statusCode >= 400) {
            throw new MagentoException($response, $statusCode);
        }

        return ($r = json_decode($response, true)) === null ? $response : $r;
    }

    public function normalizeHeaders(&$headers)
    {
        // Normalize headers
        array_walk(
            $headers,
            function (&$val, $key) {
                $key = ucfirst(strtolower($key));
                $val = ucfirst(strtolower($key)).': '.$val;
            }
        );
    }
}
