<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Exceptions\DropshipWithOpenOrdersException;
use App\Helpers;
use App\Models\Address;
use App\Models\Product;
use App\Models\ProductInventory;
use App\Models\ProductListing;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use App\Models\WarehouseLocation;
use Exception;
use Google\Service\CloudBuild\Build;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Entities\AmazonFbaReportInventory;
use Modules\Amazon\Entities\AmazonProduct;
use Throwable;

/**
 * TODO: Refactor need: 10
 *
 * Class WarehouseRepository.
 */
class WarehouseRepository extends AbstractRepository
{
    /**
     * @throws DropshipWithOpenOrdersException
     */
    public function createWarehouse(array $data): Warehouse
    {
        if (! isset($data['order_fulfillment'])) {
            $data['order_fulfillment'] = 'manual';
        }

        $warehouse = new Warehouse($data);
        $warehouse->save();
        // default warehouse location
        if (empty($data['default_location'])) {
            $data['default_location'] = ['aisle' => 'Default'];
        }
        $warehouse->defaultLocation()->create(array_merge($data['default_location'], ['is_default' => true]));

        // Create the warehouse address
        //
        $address = $this->setWarehouseAddress($warehouse, $data);

        // Update address id
        $warehouse->update([
            'address_id' => $address->id,
        ]);

        // add onto the end of the warehouse priority setting
        if (! $warehouse->isSupplierWarehouse()) {
            $warehousePriority = json_decode(Helpers::setting(Setting::KEY_WAREHOUSE_PRIORITY), true) ?: [];
            array_push($warehousePriority, $warehouse->id);
            $setting = Setting::with([])->where('key', Setting::KEY_WAREHOUSE_PRIORITY)->firstOrFail();
            $setting->value = json_encode($warehousePriority);
            $setting->save();
        }

        $warehouse->load('warehouseLocations');

        return $warehouse;
    }

    public function getPriorityStockWarehouseIdForProduct(Product $product, $quantityNeeded, $defaultToFirst = true, $orderAddress = null): ?int
    {
        [$inStockWarehouse, $firstPriorityWarehouse] = $this->getPriorityWarehouseIdFromSettings($product, $quantityNeeded, $orderAddress);

        return $inStockWarehouse ?? ($defaultToFirst ? $firstPriorityWarehouse : null);
    }


    /**
     * @throws Exception
     */
    private function getPriorityWarehouses($orderAddress = null): array
    {
        if (Setting::getValueByKey(Setting::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER) && ! empty($orderAddress) && $orderAddress['country_code'] == 'US') {
            return json_decode($this->getWarehousePriorityByLocation($orderAddress), true);
        } else {
            return json_decode(Setting::defaultWarehousePriority(), true);
        }
    }

    /**
     * @throws Exception
     */
    private function getPriorityWarehouseIdFromSettings(Product $product, $quantityNeeded, $orderAddress = null): array
    {
        $warehouses = $product->inWarehouseAvailableQuantity;

        $priorityWarehouses = $this->getPriorityWarehouses($orderAddress);

        if (! count($priorityWarehouses)) {
            return Warehouse::with([])->value('id');
        }

        foreach ($priorityWarehouses as $priorityWarehouse) {
            $warehouseRecord = Warehouse::findOrFail($priorityWarehouse);
            if ($warehouseRecord->type == Warehouse::TYPE_AMAZON_FBA) {
                $productListings = $product->productListings()
                    ->where('sales_channel_id', $warehouseRecord->integrationInstance->salesChannel->id)
                    ->whereHas('product', function ($query) use ($product) {
                        $query->where('id', $product->id);
                    })
                    ->where('document_type', AmazonProduct::class)
                    ->get();

                $productListings = $productListings->filter(function (ProductListing $productListing) use ($quantityNeeded) {
                        /** @var AmazonProduct $amazonProduct */
                        $amazonProduct = $productListing->document;
                        if ($amazonProduct->fbaInventory && $amazonProduct->fbaInventory->afn_fulfillable_quantity >= $quantityNeeded)
                        {
                            return true;
                        }
                        return false;
                    });

                if ($productListings->count() > 0) {
                    return [$priorityWarehouse, null];
                }
            }
            else {
                $warehouse = $warehouses->firstWhere('warehouse_id', $priorityWarehouse);
                if ($warehouse && $warehouse->inventory_available >= $quantityNeeded) {

                    return [$priorityWarehouse, null];
                }
            }
        }

        return [null, $priorityWarehouses[0]];
    }

    public function setWarehouseAddress(Warehouse $warehouse, array $data): Address
    {
        $address = $warehouse->address;
        if (! $address) {
            // Create new
            $address = new Address();
        }

        // Remove default location
        unset($data['default_location'], $data['type'], $data['order_fulfillment']);

        if (isset($data['order_fulfillment'])) {
            unset($data['order_fulfillment']);
        }

        $address->fill(array_merge($data, [
            //      'is_default' => true,
            'label' => Supplier::DEFAULT_ADDRESS_LABEL,
            'name' => $data['address_name'] ?? $warehouse->name,
        ]));

        $address->save();

        // Attach the address to the warehouse
        $warehouse->setAddressId($address->id);

        return $address;
    }

    /**
     * Gets priority warehouse id for product based on quantity in stock.
     *
     * @param  null  $orderAddress
     *
     * @throws Exception
     */
    public function getPriorityWarehouseIdForProduct(Product $product, $quantityNeeded, bool $defaultToFirst = true, $orderAddress = null): ?int
    {
        $warehouses = $product->inWarehouseAvailableQuantity();
        if (Setting::getValueByKey(Setting::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER) && ! empty($orderAddress) && $orderAddress['country_code'] == 'US') {
            $priorityWarehouses = json_decode($this->getWarehousePriorityByLocation($orderAddress), true);
        } else {
            $priorityWarehouses = json_decode(Setting::defaultWarehousePriority(), true);
        }
        if (! count($priorityWarehouses)) {
            return Warehouse::with([])->value('id');
        }
        foreach ($priorityWarehouses as $priorityWarehouse) {
            $warehouse = $warehouses->firstWhere('warehouse_id', $priorityWarehouse);
            if ($warehouse && $warehouse->inventory_available >= $quantityNeeded) {
                return $priorityWarehouse;
            }
        }

        if (Setting::getValueByKey(Setting::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER)) {
            return $priorityWarehouses[0];
        }

        [$inStockWarehouse, $firstPriorityWarehouse] = $this->getPriorityWarehouseIdFromSettings($product, $quantityNeeded);
        if ($inStockWarehouse) {
            return $inStockWarehouse;
        }

        // Check if product is sourced and if the supplier has stock,
        // use that warehouse (essentially dropship) - SKU-2530
        $supplier = $product->suppliers()->withPivotValue('is_default', true)->first();
        if ($supplier) {
            $inventory = SupplierInventory::with([])->where('product_id', $product->id)
                ->where('supplier_id', $supplier->id)
                ->latest()->first();
            if ($inventory && $this->isDropshipWarehouse($inventory->warehouse) && (($inventory->quantity && $inventory->quantity >= $quantityNeeded) || $inventory->in_stock)) {
                return $inventory->warehouse_id;
            }
        }

        return $defaultToFirst ? $firstPriorityWarehouse : null;
    }

    public function getWarehousesWithInventoryForProduct(Product $product, $orderAddress = null): ?array
    {
        $warehouses = $product->inWarehouseAvailableQuantity;

        if (Setting::getValueByKey(Setting::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER) && ! empty($orderAddress) && $orderAddress['country_code'] == 'US') {
            $priorityWarehouses = json_decode($this->getWarehousePriorityByLocation($orderAddress), true);
        } else {
            $priorityWarehouses = json_decode(Setting::defaultWarehousePriority(), true);
        }

        $warehousesInventory = [];

        foreach ($priorityWarehouses as $priorityWarehouse) {
            $warehouse = $warehouses->firstWhere('warehouse_id', $priorityWarehouse);

            if ($warehouse && $warehouse->inventory_available > 0) {
                $warehousesInventory[] = [

                    'warehouse_id' => $priorityWarehouse,

                    'quantity' => $warehouse->inventory_available,

                ];
            }
        }

        return $warehousesInventory;
    }

    private function distance($lat1, $lon1, $lat2, $lon2)
    {
        $theta = $lon1 - $lon2;
        $dist = sin(deg2rad($lat1)) * sin(deg2rad($lat2)) + cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * cos(deg2rad($theta));
        $dist = acos($dist);
        $dist = rad2deg($dist);
        $miles = $dist * 60 * 1.1515;
        $kilometers = $miles * 1.609344;

        return $kilometers;
    }

    public function getWarehousesWithInventory(Product $product, $quantityNeeded, $orderAddress = null): ?array
    {
        $warehouses = $product->inWarehouseAvailableQuantity;
        if (Setting::getValueByKey(Setting::KEY_PRIORITIZE_CLOSEST_WAREHOUSE_TO_CUSTOMER) && $orderAddress && $orderAddress['country_code'] == 'US') {
            $priorityWarehouses = json_decode($this->getWarehousePriorityByLocation($orderAddress), true);
        } else {
            $priorityWarehouses = json_decode(Setting::defaultWarehousePriority($orderAddress), true);
        }

        $warehousesInventory = [];
        foreach ($priorityWarehouses as $priorityWarehouse) {
            $warehouse = $warehouses->firstWhere('warehouse_id', $priorityWarehouse);
            if ($warehouse && $warehouse->inventory_available > 0) {
                $warehousesInventory[] = [
                    'warehouse_id' => $priorityWarehouse,
                    'quantity' => $warehouse->inventory_available,
                ];
            }
        }

        return $warehousesInventory;
    }

    private function isDropshipWarehouse($warehouse): bool
    {
        if (! $warehouse) {
            return false;
        }
        if ($warehouse instanceof Warehouse) {
            return $warehouse->is_dropship;
        }

        return false;
    }

    /**
     * Creates a default warehouse for the supplier.
     */
    public function createDefaultWarehouseForSupplier(Supplier $supplier): Warehouse
    {
        $warehouse = new Warehouse();
        $warehouse->supplier_id = $supplier->id;
        $warehouse->type = Warehouse::TYPE_SUPPLIER;
        $warehouse->name = Supplier::DEFAULT_WAREHOUSE_NAME;
        $warehouse->save();

        // Make warehouse default for supplier
        $supplier->setDefaultWarehouse($warehouse->id);

        // Create a default warehouse location
        $warehouse->defaultLocation()->create(['aisle' => 'Default', 'is_default' => true]);

        // Create address shell for the warehouse
        $address = new Address();
        $address->name = $supplier->name;
        $address->label = Supplier::DEFAULT_ADDRESS_LABEL;
        $address->save();

        $warehouse->setAddressId($address->id);

        return $warehouse;
    }

    /**
     * @return WarehouseLocation|mixed|null
     */
    public function getDefaultLocationForWarehouse($warehouse)
    {
        if ($warehouse instanceof Warehouse) {
            return $warehouse->defaultLocation;
        }

        $warehouse = Warehouse::with([])->findOrFail($warehouse);

        return $warehouse->defaultLocation;
    }

    public function findWarehouseByNameForSupplier($warehouseName, $supplierId): ?Warehouse
    {
        return Warehouse::with([])->where('name', $warehouseName)->where('supplier_id', $supplierId)->first();
    }

    public function getDropshippableWarehouseForProduct(Product $product): ?int
    {
        /** @var Warehouse|null $dropshippableWarehouse */
        $dropshippableWarehouse = SupplierProduct::query()
            ->selectRaw('w.id')
            ->from('supplier_products', 'sp')
            ->join('products as p', function ($join) use ($product) {
                $join->on('p.id', 'sp.product_id')
                    ->where('p.id', $product->id);
            })
            ->join('suppliers as s', 's.id', 'sp.supplier_id')
            ->join('warehouses as w', 'w.supplier_id', 's.id')
            ->where('w.dropship_enabled', 1)
            ->first();

        return $dropshippableWarehouse?->id;
    }

    public function findById($warehouseId): Warehouse|Model
    {
        return Warehouse::query()->findOrFail($warehouseId);
    }

    public function hasSingleShippableWarehouse(): bool
    {
        return Warehouse::query()
            ->whereIn('type', [Warehouse::TYPE_DIRECT, Warehouse::TYPE_3PL, Warehouse::TYPE_AMAZON_FBA])
            ->whereNull('archived_at')
            ->where('auto_routing_enabled', true)
            ->orWhere(function (Builder $query) {
                $query->where('type', Warehouse::TYPE_SUPPLIER);
                $query->whereNull('archived_at');
                $query->where('dropship_enabled', true);
            })->count() === 1;
    }

    public function getSingleShippableWarehouse(): Warehouse|Model|bool
    {
        $warehouses = Warehouse::query()
            ->whereIn('type', [Warehouse::TYPE_DIRECT, Warehouse::TYPE_3PL, Warehouse::TYPE_AMAZON_FBA])
            ->orWhere(function (Builder $query) {
                $query->where('type', Warehouse::TYPE_SUPPLIER);
                $query->where('dropship_enabled', true);
            });

        if ($warehouses->count() === 1) {
            return $warehouses->first();
        } else {
            return false;
        }
    }

    private function getWarehousePriorityByLocation(Address|array|null $orderAddress): string
    {
        $warehouses = Warehouse::query();

        $excludeTypes = [];
        if (Setting::getValueByKey(Setting::KEY_WAREHOUSE_IGNORE_SUPPLIER_TYPE)) {
            $excludeTypes[] = Warehouse::TYPE_SUPPLIER;
        }

        if (Setting::getValueByKey(Setting::KEY_WAREHOUSE_IGNORE_AMAZON_FBA_TYPE)) {
            $excludeTypes[] = Warehouse::TYPE_AMAZON_FBA;
        }

        if (! empty($excludeTypes)) {
            $warehouses->whereNotIn('type', $excludeTypes);
        }

        $warehouses->whereIn('type', [Warehouse::TYPE_DIRECT, Warehouse::TYPE_3PL]);

        $warehouses = $warehouses->whereHas('address', function ($query) {
            $query->whereNotNull('zip');
        })->get();

        // Get the latitude and longitude values for all the warehouses
        $warehousesWithCoords = [];
        foreach ($warehouses as $warehouse) {
            $warehouseAddress = $warehouse->address;
            if (empty($warehouseAddress)) {
                continue;
            }
            $zip = $warehouseAddress->zip;

            $zipData = DB::table('zip_locations')->where('zip', $zip)->first();

            // If the exact zip code is not found, get the nearest zip code
            if (! $zipData) {
                $zipData = DB::table('zip_locations')
                    ->orderByRaw("ABS(zip - $zip)")
                    ->first();
            }

            $warehouse['latitude'] = $zipData->lat;
            $warehouse['longitude'] = $zipData->lng;
            $warehousesWithCoords[] = $warehouse;
        }
        // Get the latitude and longitude values for the shipping address

        $zipCode = $orderAddress['zip'];

        $zipLocation = DB::table('zip_locations')
            ->select('zip', 'lat', 'lng')
            ->orderBy(DB::raw('ABS(zip - '.$zipCode.')'))
            ->first();

        if ($zipLocation) {
            $addressLatitude = $zipLocation->lat;
            $addressLongitude = $zipLocation->lng;
        } else {
            throw new Exception('Zip code not found');
        }

        // Find the closest warehouse to the shipping address
        $closestWarehouse = collect($warehousesWithCoords)
            ->sortBy(function ($warehouse) use ($addressLatitude, $addressLongitude) {
                return $this->distance($addressLatitude, $addressLongitude, $warehouse['latitude'],
                    $warehouse['longitude']);
            });

        return json_encode($closestWarehouse->pluck('id'));
    }

    /**
     * Get default warehouse.
     */
    public function getDefaultWarehouseForSuppliers(): Warehouse|Model
    {
        return Warehouse::query()->first();
    }

    /**
     * @throws Throwable
     */
    public function getPriorityStockWarehouseIdForSalesOrderLines(
        Collection $salesOrderLines,
        ?Address $shippingAddress = null
    ): int
    {

        $priority = $this->getPriorityWarehouses($shippingAddress);

        throw_if(empty($priority), new Exception('Warehouse priority is empty.'));

        $priorityString = implode(',', $priority);

        /** @var ProductInventory $inventory */
        $inventory = ProductInventory::query()
            ->select('warehouse_id')
            ->where('inventory_available', '>', 0)
            ->where('warehouse_id', '>', 0)
            ->whereIn('warehouse_id', $priority)
            ->whereIn('product_id', $salesOrderLines->pluck('product_id')->toArray())
            ->orderByRaw("FIELD(warehouse_id, $priorityString)")
            ->first();

        if ($inventory) {
            return $inventory->warehouse_id;
        }

        // Default to first priority warehouse.
        // Note that priority warehouse should not be empty.
        return $priority[0];
    }

    public function getWarehousesFromSupplierId(int $supplierId): EloquentCollection
    {
        return Warehouse::where('supplier_id', $supplierId)->get();
    }

    /**
     * @param  Warehouse  $warehouse
     * @return Warehouse
     */
    public function convertTo3plIfDirect(Warehouse $warehouse): Warehouse
    {
        if ($warehouse->type === Warehouse::TYPE_DIRECT) {
            $warehouse->update(['type' => Warehouse::TYPE_3PL]);
        }
        return $warehouse;
    }
}
