<?php

namespace App\Repositories;

use App\Contracts\Repositories\UserRepository as UserRepositoryContract;
use App\Data\UserData;
use App\Models\User;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

/**
 * Class UserRepository.
 */
class UserRepository extends RepositoryAbstract implements UserRepositoryContract
{
    protected string $modelClassName = 'App\User';

    public function teaserAccountUsers(): Collection
    {
        return \App\Models\User::hydrate($this->teaserAccountUsersQuery()->get()->toArray());
    }

    public function usersWithListingDiscoveryOverride($override): Collection
    {
        return \App\Models\User::hydrate($this->usersWithListingDiscoveryOverrideQuery($override)->get()->toArray());
    }

    public function usersWithIntervalOverrideQuery($override_field, $override): Collection
    {
        return DB::table('users')->where($override_field, '=', $override);
    }

    public function teaserAccountUsersQuery(): Collection
    {
        return DB::table('users')->where('is_teaser_account', 1);
    }

    protected function fixUnionQuery($query)
    {
        // Eloquent executes SELECTs combined with UNION correctly,
        // but gives you invalid SQL when calling ->toSql() on them.
        $query_sql = $query->toSql();
        if (substr(trim($query_sql), 0, 1) == '(' && strpos($query_sql, ') union (')) {
            $replace_union_with = ') AS __SUB1 UNION SELECT * FROM (';
            $end_query_with = ' AS __SUB2';
            $fixed_query_sql = str_replace(') union (', $replace_union_with, $query_sql).$end_query_with;
            $fixed_query = DB::table(DB::raw($fixed_query_sql)->getValue(DB::getQueryGrammar()))->mergeBindings($query);

            return $fixed_query;
        } else {
            return $query;
        }
    }

    public function usersNotInQuery($subquery): Builder
    {
        $subquery_sql = $subquery->toSql();
        // $fixed_subquery = $this->fixUnionSubquery($subquery_sql);
        $fixed_subquery = $subquery_sql;

        $q = DB::table('users AS u')
            ->select('u.*')
            ->whereRaw('u.id NOT IN ('.$fixed_subquery.') AND u.is_teaser_account = 0')
            ->mergeBindings($subquery);

        return $q;
    }

    public function destroyUser($id): array
    {
        $queries = [
            'announcements' => 'DELETE FROM `announcements` WHERE user_id = ?;',
            'notifications' => 'DELETE FROM `notifications` WHERE user_id = ?;',
            'api_tokens' => 'DELETE FROM `api_tokens` WHERE user_id = ?;',
            'invitations' => 'DELETE FROM `invitations` WHERE user_id = ?;',
            'invoices' => 'DELETE FROM `invoices` WHERE user_id = ?;',
            'priority_process_status' => 'DELETE FROM `priority_process_status` WHERE user_id = ?;',
            'process_status_logs' => 'DELETE FROM `process_status_logs` WHERE user_id = ?;',
            'process_status_logs_by_brand' => 'DELETE process_status_logs FROM `brands` INNER JOIN `process_status_logs` ON `process_status_logs`.`brand_id`=`brands`.`id` WHERE `brands`.`user_id` = ?;',
            'role_user' => 'DELETE FROM `role_user` WHERE user_id = ?;',
            'subscriptions' => 'DELETE FROM `subscriptions` WHERE user_id = ?;',
            'product_aliases' => 'DELETE product_aliases FROM `products` INNER JOIN `product_aliases` ON `product_aliases`.`product_id`=`products`.`id` WHERE `products`.`user_id` = ?;',
            'violation_messages' => 'DELETE violation_messages FROM `brands` INNER JOIN `violation_messages` ON `violation_messages`.`brand_id`=`brands`.`id` WHERE `brands`.`user_id` = ?;',
            'brand_aliases' => 'DELETE brand_aliases FROM `brands` INNER JOIN `brand_aliases` ON `brand_aliases`.`brand_id`=`brands`.`id` WHERE `brands`.`user_id` = ?;',
            'brand_channels' => 'DELETE brand_channels FROM `brands` INNER JOIN `brand_channels` ON `brand_channels`.`brand_id`=`brands`.`id` WHERE `brands`.`user_id` = ?;',
            'violations' => 'DELETE FROM `violations` WHERE user_id = ?;',
            'violation_notices' => 'DELETE FROM `violation_notices` WHERE user_id = ?;',
            'listings' => 'DELETE FROM `listings` WHERE user_id = ?;',
            'products' => 'DELETE FROM `products` WHERE user_id = ?;',
            'brands' => 'DELETE FROM `brands` WHERE user_id = ?;',
            'team_memberships' => 'DELETE FROM `team_users` WHERE user_id = ?;',
            'team_subscriptions' => 'DELETE team_subscriptions FROM `teams` INNER JOIN `team_subscriptions` ON `team_subscriptions`.`team_id`=`teams`.`id` WHERE `teams`.`owner_id` = ?;',
            'team_users' => 'DELETE team_users FROM `teams` INNER JOIN `team_users` ON `team_users`.`team_id`=`teams`.`id` WHERE `teams`.`owner_id` = ?;',
            'teams' => 'DELETE FROM `teams` WHERE `owner_id` = ?;',
            'users' => 'DELETE FROM `users` WHERE `id` = ?;',
        ];
        $deleted = [];
        foreach ($queries as $k => $query) {
            try {
                $deleted[$k] = DB::delete($query, [$id]);
            } catch (\Exception $e) {
                \Debugbar::error($e->getMessage());
            }
        }
        \Debugbar::info($deleted);

        return $deleted;
    }

    public function getAdminUsers(): EloquentCollection
    {
        return User::where('is_admin', true)->get();
    }

    public function save(UserData $userData): Model
    {
        return User::create($userData->toArray());
    }

    public function findBy($field, $value): ?Model
    {
        return User::where($field, $value)->first();
    }
}
