<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Collections\SupplierCollection;
use App\Models\PurchaseOrder;
use App\Models\Supplier;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;

class SupplierRepository extends AbstractRepository
{
    /**
     * @param  SupplierCollection  $data
     * @param  bool  $insert
     * @param  bool  $update
     * @return array
     */
    public function saveBulk(SupplierCollection $data, bool $insert, bool $update): array
    {
        $defaultWarehouse = app(WarehouseRepository::class)->getDefaultWarehouseForSuppliers();

        $tempTable = Supplier::dataFeedBulkImport([
            'data_to_import' => [
                'type' => 'json',
                'data' => $data->toJson(),
            ],
            'insert' => $insert,
            'update' => $update,
            'mappings' => [
                [
                    'expected_column_name' => 'name',
                    'data_column_name' => 'name',
                    'expected_column_type' => 'string',
                ],
                [
                    'expected_column_name' => 'timezone',
                    'data_column_name' => 'timezone',
                    'expected_column_type' => 'string',
                ]
            ],
            'default_columns' => [
                [
                    'expected_column_name' => 'created_at',
                    'default_value' => now(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'updated_at',
                    'default_value' => now(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'default_warehouse_id',
                    'default_value' => $defaultWarehouse->id,
                    'expected_column_type' => 'integer',
                ],
            ],
            'unique_by_columns' => [
                'name',
            ],
        ]);

        return [
            'temp_table' => $tempTable,
            'status' => true,
        ];
    }

    /**
     * @return EloquentCollection
     */
    public function getSuppliersEligibleForBulk(): EloquentCollection
    {
        return Supplier::with(['purchaseOrders'])
            ->whereHas('purchaseOrders', function (Builder $builder) {
                $builder->where('submission_status', PurchaseOrder::SUBMISSION_STATUS_QUEUED);
            })
            ->where('po_processing_method', Supplier::PO_DELIVERY_METHOD_BATCH)
            ->where('auto_fulfill_dropship', true)
            ->whereNotNull('po_batch_schedule')
            ->get();
    }

    /**
     * @return EloquentCollection
     */
    public function getSuppliersEligibleForBackorderPurchasing(): EloquentCollection
    {
        return Supplier::query()
            ->where('auto_generate_backorder_po', true)
            ->whereNotNull('backorder_po_schedule')
            ->get();
    }

    /**
     * @param  array  $supplierIds
     * @return array
     */
    public function getDefaultCurrenciesForSuppliers(array $supplierIds): array
    {
        return Supplier::query()
            ->select('suppliers.id', 'currencies.id as currency_id', 'currencies.conversion')
            ->join('supplier_to_pricing_tier as stpt', 'stpt.supplier_id', '=', 'suppliers.id')
            ->join('supplier_pricing_tiers as spt', 'spt.id', '=', 'stpt.supplier_pricing_tier_id')
            ->join('currencies', 'currencies.code', '=', 'spt.currency_code')
            ->where('spt.is_default', true)
            ->whereIn('suppliers.id', $supplierIds)
            ->get(['suppliers.id', 'currency_id', 'conversion'])
            ->toArray();
    }
}
