<?php

namespace App\Repositories\Shopify;

use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Shopify\ShopifyWebhook;
use App\Models\Shopify\ShopifyWebhookEvent;
use App\Services\Shopify\Orders\Actions\ShopifyDownloadOrder;
use Carbon\Carbon;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

class ShopifyWebhookEventRepository
{
    public function save(IntegrationInstance $integrationInstance, string $topic, array $data): void
    {
        $unique_id = null;
        $payload_date = null;

        switch ($topic) {
            case ShopifyWebhook::TOPIC_ORDERS_CREATE:
            case ShopifyWebhook::TOPIC_ORDERS_UPDATED:
                $topic = ShopifyWebhookEvent::TOPIC_ORDERS;
                $unique_id = $data['id'];
                $payload_date = $data['updated_at'] ?? $data['created_at'];
                break;
            case ShopifyWebhook::TOPIC_ORDER_TRANSACTIONS_CREATE:

                $unique_id = $data['id'];
                $payload_date = $data['processed_at'];
                break;
            default:
                break;
        }

        $webhookEvent = new ShopifyWebhookEvent([
            'integration_instance_id' => $integrationInstance->id,
            'topic' => $topic,
            'unique_id' => $unique_id,
            'payload_date' => Carbon::parse($payload_date),
            'json_data' => json_encode($data, JSON_UNESCAPED_SLASHES),
        ]);

        $webhookEvent->save();
    }

    public function purgeOld(IntegrationInstance $integrationInstance)
    {
        ShopifyWebhookEvent::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNotNull('processed_at')
            ->where('processed_at', '<=', Carbon::now()->subDays(7))
            ->delete();
    }

    private function purgeObsolete(IntegrationInstance $integrationInstance): void
    {
        $tmpTableName = 'swe_ordered'.uniqid();

        /*
         * From unprocessed records, we want to only keep the latest payload_date for each unique_id/integration_instances_id/topic
         */
        DB::statement('
            CREATE TEMPORARY TABLE '.$tmpTableName.' (
               SELECT
                 id,
                 unique_id,
                 payload_date,
                 processed_at,
                 topic,
                   RANK() OVER (
                     PARTITION BY
                       integration_instance_id,
                       unique_id,
                       topic
                     ORDER BY payload_date DESC
                   ) AS rank
               FROM shopify_webhook_events
               WHERE processed_at IS NULL
               AND integration_instance_id = '.$integrationInstance->id.'
            )');

        customlog('shopifyNeedsUpdating', 'Purging started');
        DB::statement('
            DELETE swe
            FROM shopify_webhook_events AS swe
            INNER JOIN (
                SELECT swe_ordered.id
                FROM ' . $tmpTableName . ' AS swe_ordered
                WHERE swe_ordered.rank > 1
                OR EXISTS (
                    SELECT 1 
                    FROM shopify_webhook_events AS s2
                    WHERE s2.processed_at IS NOT NULL
                    AND s2.integration_instance_id = ' . $integrationInstance->id . '
                    AND s2.unique_id = swe_ordered.unique_id
                    AND s2.topic = swe_ordered.topic
                    AND s2.payload_date = swe_ordered.payload_date)
            ) AS subquery
            ON swe.id = subquery.id
            WHERE swe.integration_instance_id = ' . $integrationInstance->id);
        customlog('shopifyNeedsUpdating', 'Purging complete');
    }

    public function getWebhookEventsNeedingProcessing(IntegrationInstance $integrationInstance): Collection
    {
        return ShopifyWebhookEvent::query()
            ->fromSub(function ($query) {
                $query->select('unique_id', DB::raw('MAX(payload_date) as max_payload_date'))
                    ->from('shopify_webhook_events')
                    ->whereNull('processed_at')
                    ->groupBy('unique_id');
            }, 'sub')
            ->join('shopify_webhook_events', function ($join) {
                $join->on('shopify_webhook_events.unique_id', '=', 'sub.unique_id')
                    ->on('shopify_webhook_events.payload_date', '=', 'sub.max_payload_date');
            })
            ->where('shopify_webhook_events.integration_instance_id', $integrationInstance->id)
            ->whereNull('shopify_webhook_events.processed_at')
            ->whereNotExists(function ($query) {
                $query->select(DB::raw(1))
                    ->from('shopify_webhook_events as swe2')
                    ->whereColumn('swe2.unique_id', 'shopify_webhook_events.unique_id')
                    ->where('swe2.processed_at', '!=', null)
                    ->whereRaw('swe2.payload_date > shopify_webhook_events.payload_date');
            })
            ->get();
    }

    /**
     * @throws BindingResolutionException
     * @throws Throwable
     */
    public function process(IntegrationInstance $integrationInstance): void
    {
        $this->purgeObsolete($integrationInstance);

        customlog('shopifyNeedsUpdating', 'Selecting webhook events to process');
        $webhookEvents = $this->getWebhookEventsNeedingProcessing($integrationInstance);
        customlog('shopifyNeedsUpdating', count($webhookEvents).' Webhook events found');

        $orders = [];
        $order_transactions = [];
        $webhookEventsForOrders = [];

        /** @var ShopifyWebhookEvent $webhookEvent */
        foreach ($webhookEvents as $webhookEvent) {
            switch ($webhookEvent->topic) {
                case ShopifyWebhookEvent::TOPIC_ORDERS:
                    $orders[] = json_decode($webhookEvent->json_data, 1);
                    $webhookEventsForOrders[] = $webhookEvent;
                    break;
                case ShopifyWebhook::TOPIC_ORDER_TRANSACTIONS_CREATE:
                    $order_transactions[] = json_decode($webhookEvent->json_data, 1);
                    break;
                default:
                    break;
            }
        }

        if (! empty($webhookEventsForOrders)) {
            $this->processWebhookEventsForOrders($integrationInstance, $webhookEventsForOrders, $orders);
        }

        if (! empty($order_transactions)) {
            $this->processForOrderTransactions($integrationInstance, $order_transactions);
        }
    }

    private function processWebhookEventsForOrders(IntegrationInstance $integrationInstance, array $webhookEvents, array $orders): void
    {
        customlog('shopifyNeedsUpdating', 'Updating '.count($webhookEvents).' orders for '.$integrationInstance->id, array_map(function ($data) {
            return [
                'id' => $data->id,
                'unique_id' => $data->unique_id,
                'payload_date' => $data->payload_date,
            ];
        }, $webhookEvents));
        ShopifyDownloadOrder::bulkSaveShopifyOrderToSkuDB($integrationInstance->id, $orders, ShopifyOrder::DOWNLOADED_BY_WEBHOOK);
        $this->markIdsAsProcessed($integrationInstance, array_column($webhookEvents, 'id'));
    }

    /**
     * @throws BindingResolutionException
     */
    public function processForOrderTransactions(IntegrationInstance $integrationInstance, $transactions): void
    {
        $savedIds = app()->make(ShopifyTransactionRepository::class)->bulkSave($integrationInstance, $transactions, true);
        $this->markAsProcessed($integrationInstance, $savedIds);
    }

    /**
     * @throws BindingResolutionException
     */
    public function reprocessUnregisteredTransactions(IntegrationInstance $integrationInstance)
    {
        $transactions = DB::select('
            SELECT swe.json_data
            FROM shopify_webhook_events AS swe
            INNER JOIN shopify_orders AS so
            ON so.shopify_order_id = JSON_EXTRACT(swe.json_data, "$.order_id") AND so.integration_instance_id = swe.integration_instance_id
            WHERE so.transactions_updated_at IS NULL;
            ');

        $transactions = array_map(function ($value) {
            return json_decode($value->json_data, 1);
        }, $transactions);

        $this->processForOrderTransactions($integrationInstance, $transactions);
    }

    private function markAsProcessed(IntegrationInstance $integrationInstance, array $unique_ids): void
    {
        ShopifyWebhookEvent::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereIn('unique_id', $unique_ids)
            ->whereNull('processed_at')
            ->update([
                'processed_at' => Carbon::now(),
            ]);
    }

    private function markIdsAsProcessed(IntegrationInstance $integrationInstance, array $ids): void
    {
        ShopifyWebhookEvent::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereIn('id', $ids)
            ->update([
                'processed_at' => Carbon::now(),
            ]);
    }
}
