<?php

namespace App\Repositories\Shopify;

use App\Integrations\Shopify;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyOrder;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;

class ShopifyTransactionRepository
{
    public function save(IntegrationInstance $integrationInstance, int $shopify_order_id, array $data)
    {
        /** @var ShopifyOrder $shopifyOrder */
        $shopifyOrder = ShopifyOrder::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->where('shopify_order_id', $shopify_order_id)
            ->firstOrFail();

        $shopifyOrder->transactions_updated_at = Carbon::now();
        $shopifyOrder->transactions = $data;

        $shopifyOrder->update();
    }

    public function getOrderIdsForTransactionsNeedingUpdate(IntegrationInstance $integrationInstance): array
    {
        return ShopifyOrder::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->where('total_price', '!=', 0)
            ->where(function (Builder $query) {
                $query->whereNull('transactions_updated_at');
                $query->orWhereRaw('transactions_updated_at < sku_updated_at');
            })
            ->pluck('shopify_order_id')->toArray();
    }

    public function getOldestUpdatedAtDateForTransactionsNeedingUpdate(IntegrationInstance $integrationInstance): ?string
    {
        return ShopifyOrder::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->where('total_price', '!=', 0)
            ->where(function (Builder $query) {
                $query->whereNull('transactions_updated_at');
                $query->orWhereRaw('transactions_updated_at < sku_updated_at');
            })
            ->min('updated_at');
    }

    public function getNewestUpdatedAtDateForTransactionsNeedingUpdate(IntegrationInstance $integrationInstance): ?string
    {
        return ShopifyOrder::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->where('total_price', '!=', 0)
            ->where(function (Builder $query) {
                $query->whereNull('transactions_updated_at');
                $query->orWhereRaw('transactions_updated_at < sku_updated_at');
            })
            ->max('updated_at');
    }

    public function bulkSave(IntegrationInstance $integrationInstance, array $transactions, bool $rest_payload = false): array
    {
        $savedTransactionIds = [];
        foreach ($transactions as $transaction) {
            /** @var ShopifyOrder $shopifyOrder */
            $shopifyOrder = ShopifyOrder::query()
                ->where('integration_instance_id', $integrationInstance->id)
                ->where('shopify_order_id', $transaction['order_id'])
                ->first();
            if (! $shopifyOrder) {
                customlog('shopifyNeedsUpdating', 'Transaction data came in for order that is not yet saved, so skipping');

                // Somehow the transaction data came in for an order that is not yet saved, so skip
                continue;
            }

            if ($rest_payload) {
                $transaction = $this->mapRestPayloadToGraphQLPayload($transaction);
            }

            $transactions = collect($shopifyOrder->transactions ?: [])->keyBy('id')->put($transaction['id'], $transaction);

            $shopifyOrder->transactions = $transactions->values()->toArray();
            $shopifyOrder->transactions_updated_at = Carbon::now();
            $shopifyOrder->update();

            $savedTransactionIds[] = Shopify::getIdFromGID($transaction['id']);
        }

        return $savedTransactionIds;
    }

    public function mapRestPayloadToGraphQLPayload($restTransactionPayload): array
    {
        return [
            'id' => "gid://shopify/OrderTransaction/{$restTransactionPayload['id']}",
            'kind' => mb_strtoupper($restTransactionPayload['kind']),
            'status' => mb_strtoupper($restTransactionPayload['status']),
            // fees, gateway and receipt are used to extract payment cost
            'fees' => [], // TODO: we can extract it from receipt.metadata.transaction_fee_total_amount, but it does not return with a decimal point!, it's only for shopify_payments gateway
            'gateway' => $restTransactionPayload['gateway'],
            'receipt' => $restTransactionPayload['receipt'], // TODO: in rest, it returns as json, but in graphql it returns in a special format
            'processedAt' => $restTransactionPayload['processed_at'],
            'amount' => $restTransactionPayload['amount'],
            'amountSet' => [
                'presentmentMoney' => [
                    'amount' => $restTransactionPayload['amount'],
                    'currencyCode' => $restTransactionPayload['currency'],
                ],
            ],
        ];
    }
}
