<?php

namespace App\Repositories\Shopify;

use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\Shopify\ShopifyBulkOperation;
use App\Models\Shopify\ShopifyProduct as ShopifyProduct;
use App\Notifications\MonitoringMessage;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Throwable;

class ShopifyProductRepository
{
    public function deleteProductListingsRemovedFromShopify(IntegrationInstance $integrationInstance): void
    {
        $variant_ids = ShopifyProduct::query()
            ->select('shopify_products.variant_id')
            ->join('product_listings', 'product_listings.document_id', '=', 'shopify_products.id')
            ->where('shopify_products.integration_instance_id', $integrationInstance->id)
            ->where('shopify_products.removed_from_shopify', 1)
            ->where('product_listings.document_type', ShopifyProduct::class)
            ->get()->pluck('variant_id')->flatten()->toArray();

        //        if ($deleteCount = count($variant_ids))
        //        {
        //            Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage('The following ' . $deleteCount . ' product listings will be removed for ' . $integrationInstance->name . ':' . implode(',', $variant_ids)));
        //        }

        ProductListing::query()
            ->join('shopify_products', 'shopify_products.id', '=', 'product_listings.document_id')
            ->where('shopify_products.integration_instance_id', $integrationInstance->id)
            ->where('shopify_products.removed_from_shopify', 1)
            ->where('product_listings.document_type', ShopifyProduct::class)
            ->each(function (ProductListing $productListing) {
                $productListing->delete();
            });
    }

    public function markProductVariantsRemovedThatExistInSkuButMissingFromShopify(IntegrationInstance $integrationInstance, array $shopifyVariantIds)
    {
        $tempTableName = 'tempShopifyProducts_'.Str::random(10);

        $sql_temp_table = "CREATE TEMPORARY TABLE $tempTableName
            (`id` bigint(20) unsigned, PRIMARY KEY (`id`)) ENGINE=InnoDB;";

        $sql_insert = 'INSERT INTO '.$tempTableName.' (id) VALUES ('.implode('),(', $shopifyVariantIds).');';

        $sql_select = "SELECT sku FROM shopify_products AS sp LEFT JOIN $tempTableName AS tsp 
            ON tsp.id = sp.variant_id
            WHERE tsp.id IS NULL AND sp.integration_instance_id = ".$integrationInstance->id;

        $sql_update = "UPDATE shopify_products AS sp LEFT JOIN $tempTableName AS tsp 
            ON tsp.id = sp.variant_id 
            SET sp.removed_from_shopify = 1
            WHERE tsp.id IS NULL AND sp.integration_instance_id = ".$integrationInstance->id;

        DB::statement($sql_temp_table);
        DB::statement($sql_insert);

        $skusToRemove = collect(DB::select($sql_select))->pluck('sku');

        if ($skusToRemove->count() > 0) {
            slack('The following '.$skusToRemove->count().' product variants will be marked as removed from shopify for '.$integrationInstance->name.':'.$skusToRemove->implode(', '));
        }

        DB::update($sql_update);
    }

    public function markProductVariantsActiveFromShopify(IntegrationInstance $integrationInstance, array $shopifyVariantIds)
    {
        $tempTableName = 'tempShopifyProducts_'.Str::random(10);

        $sql_temp_table = "CREATE TEMPORARY TABLE $tempTableName
            (`id` bigint(20) unsigned, PRIMARY KEY (`id`)) ENGINE=InnoDB;";

        $sql_insert = "INSERT INTO $tempTableName (id) VALUES (".implode('),(', $shopifyVariantIds).');';

        $sql_update = "UPDATE shopify_products AS sp INNER JOIN $tempTableName AS tsp 
            ON tsp.id = sp.variant_id 
            SET sp.removed_from_shopify = 0 AND sp.integration_instance_id = ".$integrationInstance->id;

        DB::statement($sql_temp_table);
        DB::statement($sql_insert);
        DB::update($sql_update);
    }

    /**
     * @throws Throwable
     */
    public function getProductVariationIdsFromJSONL(ShopifyBulkOperation $bulkOperation): array
    {
        $order_id_prefix_pattern = '/^.*gid:\\\\\/\\\\\/shopify\\\\\/ProductVariant\\\\\/([0-9]*).*$/';

        $bulkOperation->reformatted_filename = 'import-'.$bulkOperation->filename;

        $fh_read = fopen(Storage::disk('shopify')->path('').$bulkOperation->filename, 'r');

        $productVariationIds = [];

        throw_if(! $fh_read, new Exception('Error opening file for bulk operation: '.$bulkOperation->filename));

        while (($line = fgets($fh_read)) !== false) {
            if (substr($line, 24, 14) == 'ProductVariant') {
                $matches = [];
                if (preg_match($order_id_prefix_pattern, substr($line, 5), $matches) === 0) {
                    throw new Exception('Error parsing line of JSONL while trying to find product variant id');
                }
                $productVariationIds[] = $matches[1];
            } else {
                throw new Exception('Unexpected line');
            }
        }
        fclose($fh_read);

        return $productVariationIds;
    }
}
