<?php

namespace App\Repositories\Shopify;

use App\Models\SalesCredit;
use App\Models\SalesOrderFulfillment;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Shopify\ShopifyOrderMapping;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class ShopifyOrderMappingRepository
{
    public function create(array $attributes): ShopifyOrderMapping
    {
        return ShopifyOrderMapping::create($attributes);
    }

    public function upsert(array $attributes, array $values): ShopifyOrderMapping
    {
        return ShopifyOrderMapping::updateOrCreate($attributes, $values);
    }

    public function getRefundMapByLineItemId(ShopifyOrder $shopifyOrder, string $refundId, string $lineItemId): ?ShopifyOrderMapping
    {
        return $shopifyOrder
            ->orderMappings()
            ->where('link_type', ShopifyOrderMapping::LINK_TYPE_REFUNDS)
            ->where('link_id', $refundId)
            ->where('line_id', $lineItemId)
            ->first();
    }

    public function getMappingsForLinks(string $linkType, int|array $linkIds): EloquentCollection
    {
        return ShopifyOrderMapping::where('link_type', $linkType)
            ->whereIn('link_id', Arr::wrap($linkIds))
            ->get();
    }

    public function getMappingsForTypeForShopifyOrder(ShopifyOrder $shopifyOrder, string $linkType): EloquentCollection
    {
        return $shopifyOrder
            ->orderMappings()
            ->where('link_type', $linkType)
            ->get();
    }

    public function bulkDeleteForShopifyOrders(array|Collection $shopifyOrderIds): void
    {
        ShopifyOrderMapping::whereIn('shopify_order_id', $shopifyOrderIds)
            ->delete();
    }

    public function getMappingForTypeForShopifyOrder(ShopifyOrder $shopifyOrder, string $linkType, int $skuLinkId): ?ShopifyOrderMapping
    {
        return $shopifyOrder
            ->orderMappings()
            ->where('link_type', $linkType)
            ->where('sku_link_id', $skuLinkId)
            ->first();
    }

    public function getTotalFulfillmentUnitsProcessed(int $fulfillmentId)
    {
        return ShopifyOrderMapping::where('link_type', ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS)
            ->where('link_id', $fulfillmentId)
            ->sum('quantity');
    }

    public function getSalesCreditIdsForShopifyOrder(ShopifyOrder $shopifyOrder)
    {
        return $shopifyOrder
            ->orderMappings
            ->where('link_type', ShopifyOrderMapping::LINK_TYPE_REFUNDS)
            ->pluck('sku_link_id')
            ->unique();
    }

    public function getShopifyFulfillmentIdForSalesOrderFulfillment(SalesOrderFulfillment $salesOrderFulfillment)
    {
        return ShopifyOrderMapping::where('link_type', ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS)
            ->firstWhere('sku_link_id', $salesOrderFulfillment->id)['link_id'] ?? null;
    }

    public function getFulfillmentIdsForShopifyOrder(ShopifyOrder $shopifyOrder)
    {
        return $shopifyOrder
            ->orderMappings()
            ->where('link_type', ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS)
            ->pluck('sku_link_id')
            ->unique();
    }

    public function deleteWithSiblings(ShopifyOrderMapping $shopifyOrderMapping): void
    {
        // Delete with any siblings
        ShopifyOrderMapping::where('shopify_order_id', $shopifyOrderMapping->shopify_order_id)
            ->where('link_type', $shopifyOrderMapping->link_type)
            ->where('link_id', $shopifyOrderMapping->link_id)
            ->where('sku_link_type', $shopifyOrderMapping->sku_link_type)
            ->where('sku_link_id', $shopifyOrderMapping->sku_link_id)
            ->delete();
    }

    public function deleteMappingsForSalesCredit(SalesCredit $salesCredit): void
    {
        // Delete any shopify order mappings
        /** @var ShopifyOrderMapping $shopifyOrderMapping */
        ShopifyOrderMapping::with([])
            ->where('link_type', ShopifyOrderMapping::LINK_TYPE_REFUNDS)
            ->where('sku_link_id', $salesCredit->id)
            ->each(function (ShopifyOrderMapping $shopifyOrderMapping) {
                // Delete the mapping for the sales credit.
                // We also delete all cancelations for the same
                // shopify order
                $shopifyOrderMapping->delete();

                ShopifyOrderMapping::where('shopify_order_id', $shopifyOrderMapping->shopify_order_id)
                    ->where('link_type', ShopifyOrderMapping::LINK_TYPE_REFUNDS)
                    ->where('link_id', $shopifyOrderMapping->link_id)
                    ->where('sku_link_type', ShopifyOrderMapping::SKU_LINK_TYPE_CANCELLATION)
                    ->delete();
            });
    }

    private function getMappingsForShopifyOrderQuery(ShopifyOrder $shopifyOrder): Builder|ShopifyOrderMapping
    {
        return ShopifyOrderMapping::where('shopify_order_id', $shopifyOrder->id);
    }

    public function deleteMappingsForShopifyOrder(ShopifyOrder $shopifyOrder): void
    {
        $this->getMappingsForShopifyOrderQuery($shopifyOrder)->delete();
    }

    public function deleteRefundMappingsForShopifyOrder(ShopifyOrder $shopifyOrder): void
    {
        $this->getMappingsForShopifyOrderQuery($shopifyOrder)
            ->where('link_type', ShopifyOrderMapping::LINK_TYPE_REFUNDS)
            ->delete();
    }

    public function deleteMappingsForSalesOrderFulfillment(SalesOrderFulfillment $salesOrderFulfillment): void
    {
        ShopifyOrderMapping::with([])
            ->where('link_type', ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS)
            ->where('sku_link_id', $salesOrderFulfillment->id)
            ->delete();
    }

    public function shopifyOrderIsMappedToFulfillmentId(ShopifyOrder $shopifyOrder, int $fulfillmentId): bool
    {
        return $shopifyOrder->orderMappings
            ->where('link_type', ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS)
            ->where('link_id', $fulfillmentId)
            ->isNotEmpty();
    }

    /*
    |--------------------------------------------------------------------------
    | Utility methods
    |--------------------------------------------------------------------------
    */

    public function fixLineTypeForFulfillments(): void
    {
        // store line type for fulfillment mappings
        ShopifyOrderMapping::query()->whereNull('line_type')
            ->where('link_type', ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS)
            ->update(['line_type' => ShopifyOrderMapping::LINE_TYPE_LINE_ITEM]);
    }

    public function fixLineTypeAndLinkTypeForRefundAdjustments(): void
    {
        // fix storing link type and line type for refund adjustments
        ShopifyOrderMapping::query()->whereNull('line_type')
            ->where('link_type', 'refund_adjustments')
            ->update(['line_type' => ShopifyOrderMapping::LINE_TYPE_REFUND_ADJUSTMENT,
                'link_type' => ShopifyOrderMapping::LINK_TYPE_REFUNDS]);
    }

    public function remapRefundMappingsForCancellations(): void
    {
        // remap refund mappings
        ShopifyOrderMapping::with('shopifyOrder')
            ->whereNull('line_type')
            ->where('link_type', 'refunds')
            ->where('sku_link_type', '!=', ShopifyOrderMapping::SKU_LINK_TYPE_CANCELLATION)
            ->groupBy(['shopify_order_id', 'link_type', 'link_id', 'sku_link_id', 'sku_link_type'])
            ->select(['shopify_order_id', 'link_type', 'link_id', 'sku_link_id', 'sku_link_type'])
            ->eachById(function (ShopifyOrderMapping $mapping) {
                DB::transaction(function () use ($mapping) {
                    // delete old mapping
                    ShopifyOrderMapping::query()->where(
                        [
                            'shopify_order_id' => $mapping->shopify_order_id,
                            'link_type' => $mapping->link_type,
                            'link_id' => $mapping->link_id,
                        ])->delete();
                    // map again with new lines
                    $mapping->shopifyOrder->markRefundAsProcessed($mapping->link_id, $mapping->sku_link_id, $mapping->sku_link_type);
                });
            }, 1000, 'shopify_order_id');
    }

    public function remapCancellationMappings(): void
    {
        // remap cancellation mappings
        ShopifyOrderMapping::with('shopifyOrder')
            ->whereNull('line_type')
            ->where('link_type', 'refunds')
            ->where('sku_link_type', ShopifyOrderMapping::SKU_LINK_TYPE_CANCELLATION)
            ->groupBy(['shopify_order_id', 'link_type', 'link_id'])
            ->select(['shopify_order_id', 'link_type', 'link_id'])
            ->eachById(function (ShopifyOrderMapping $mapping) {
                DB::transaction(function () use ($mapping) {
                    // delete old mapping
                    ShopifyOrderMapping::query()->where(
                        [
                            'shopify_order_id' => $mapping->shopify_order_id,
                            'link_type' => $mapping->link_type,
                            'link_id' => $mapping->link_id,
                        ])->delete();

                    $refund = collect($mapping->shopifyOrder->refunds)->firstWhere('id', $mapping->link_id);
                    $mapping->shopifyOrder->handleAdjustmentRefund($refund);
                });
            }, 1000, 'shopify_order_id');
    }

    public function remapRefundMappingsForSalesCreditLineRefunds(): void
    {
        ShopifyOrderMapping::with('shopifyOrder')
            ->where('line_type', ShopifyOrderMapping::LINE_TYPE_LINE_ITEM)
            ->where('link_type', 'refunds')
            ->where(function ($query) {
                $query->where('sku_link_type', 'not like', '%'.\App\Models\SalesCreditLine::class.'%')
                    ->orWhereNull('sku_link_type');
            })
            ->select(['shopify_order_id', 'link_type', 'link_id', 'sku_link_id', 'sku_link_type'])
            ->eachById(function (ShopifyOrderMapping $mapping) {
                DB::transaction(function () use ($mapping) {
                    // delete old mapping
                    ShopifyOrderMapping::query()->where(
                        [
                            'shopify_order_id' => $mapping->shopify_order_id,
                            'link_type' => $mapping->link_type,
                            'link_id' => $mapping->link_id,
                        ])->delete();
                    // map again with new lines
                    $mapping->shopifyOrder->handleRefunds();
                });
            }, 1000, 'shopify_order_id');
    }

    public function remapRefundMappingsForSalesCreditLineRefundAdjustments(): void
    {
        ShopifyOrderMapping::with('shopifyOrder')
            ->where('line_type', ShopifyOrderMapping::LINE_TYPE_REFUND_ADJUSTMENT)
            ->where('link_type', 'refunds')
            ->where(function ($query) {
                $query->where('sku_link_type', 'not like', \App\Models\SalesCreditLine::class)
                    ->orWhereNull('sku_link_type');
            })
            ->groupBy(['shopify_order_id', 'link_type', 'link_id'])
            ->select(['shopify_order_id', 'link_type', 'link_id'])
            ->eachById(function (ShopifyOrderMapping $mapping) {
                DB::transaction(function () use ($mapping) {
                    // delete old mapping
                    ShopifyOrderMapping::query()->where(
                        [
                            'shopify_order_id' => $mapping->shopify_order_id,
                            'link_type' => $mapping->link_type,
                            'link_id' => $mapping->link_id,
                        ])->delete();

                    $refund = collect($mapping->shopifyOrder->refunds)->firstWhere('id', $mapping->link_id);

                    if (is_array($refund)) {
                        $mapping->shopifyOrder->handleAdjustmentRefund($refund);
                    }
                });
            }, 1000, 'shopify_order_id');
    }
}
