<?php

namespace App\Repositories\Shopify;

use App\Enums\Shopify\ShopifyBulkOperationStatusEnum;
use App\Enums\Shopify\ShopifyBulkOperationTypeEnum;
use App\Integrations\Shopify;
use App\Jobs\Shopify\ProcessShopifyOrderTransactionsJSONLJob;
use App\Jobs\Shopify\ProcessShopifyProductVariantsBulkOperationJob;
use App\Jobs\Shopify\ReformatShopifyOrderTransactionsJSONLJob;
use App\Models\IntegrationInstance;
use App\Models\Shopify\ShopifyBulkOperation;
use Carbon\Carbon;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Storage;
use PHPShopify\Exception\ApiException;
use PHPShopify\Exception\CurlException;
use Throwable;

class ShopifyBulkOperationRepository
{
    public function create(IntegrationInstance $integrationInstance, array $data): bool|ShopifyBulkOperation
    {
        if (ShopifyBulkOperation::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->where('type', $data['type'])
            ->where('status', '!=', ShopifyBulkOperationStatusEnum::STATUS_COMPLETED())
            ->first()) {
            //Log::info('You cannot start another Shopify bulk operation until the existing one has finished for '.$data['type'].':'.$data['operation_type']);

            return false;
        }

        //Log::debug('Saving bulk operation', $data);

        $bulkOperation = new ShopifyBulkOperation(array_merge($data, [
            'integration_instance_id' => $integrationInstance->id,
            'object_count' => 0,
            'file_size' => 0,
        ]));

        $bulkOperation->save();

        return $bulkOperation;

        /*
         * At this point you should have webhooks in place to know when the next step should be done (update)
         */
    }

    /**
     * @throws ApiException
     * @throws Throwable
     * @throws CurlException
     */
    public function update(IntegrationInstance $integrationInstance, array $data): void
    {
        /** @var ShopifyBulkOperation $bulkOperation */
        $bulkOperation = ShopifyBulkOperation::query()->where('admin_graphql_api_id', $data['admin_graphql_api_id'])->firstOrFail();

        $bulkOperation->integration_instance_id = $integrationInstance->id;
        $bulkOperation->admin_graphql_api_id = $data['admin_graphql_api_id'];
        $bulkOperation->completed_at = Carbon::parse($data['completed_at'])->setTimezone('UTC');
        $bulkOperation->created_at = Carbon::parse($data['created_at'])->setTimezone('UTC');
        $bulkOperation->error_code = $data['error_code'];
        $bulkOperation->type = $data['type'];
        $bulkOperation->token = @$data['token'];
        $bulkOperation->status = strtoupper($data['status']);

        $bulkOperation->update();

        if ($bulkOperation->completed_at) {
            $this->getDataURL($bulkOperation, $integrationInstance);
        }
    }

    /**
     * @throws ApiException
     * @throws CurlException
     * @throws \Exception
     * @throws Throwable
     */
    public function getDataURL(ShopifyBulkOperation $bulkOperation, IntegrationInstance $integrationInstance): void
    {
        $shopify = new Shopify($integrationInstance);

        $response = $shopify->getBulkOperationDataURL($bulkOperation->admin_graphql_api_id);

        $bulkOperation->url = $response['data']['node']['url'];
        $bulkOperation->object_count = $response['data']['node']['objectCount'];
        $bulkOperation->file_size = $response['data']['node']['fileSize'];
        $bulkOperation->update();

        if ($this->downloadData($bulkOperation)) {
            //Log::debug("Download for {$bulkOperation->operation_type->value} successful");

            switch ($bulkOperation->operation_type) {
                case ShopifyBulkOperationTypeEnum::OPERATION_TYPE_GET_ORDERS:
                    /*
                     * Disabling for now until the data issues are sorted
                     */
                    //ReformatShopifyOrderJSONLJob::dispatch();
                    break;
                case ShopifyBulkOperationTypeEnum::OPERATION_TYPE_GET_ORDER_TRANSACTIONS:
                    Bus::batch([
                        [
                            new ReformatShopifyOrderTransactionsJSONLJob($bulkOperation),
                            new ProcessShopifyOrderTransactionsJSONLJob($bulkOperation),
                        ],
                    ])
                        ->name('Download/Process Transactions on '.$integrationInstance->name.' (ID: '.$bulkOperation->id.')')
                        ->onQueue('salesOrderProcessing')
                        ->dispatch();
                    break;
                case ShopifyBulkOperationTypeEnum::OPERATION_TYPE_GET_PRODUCT_VARIANTS:
                    dispatch(new ProcessShopifyProductVariantsBulkOperationJob($bulkOperation))->onQueue('sales-channels');
                    break;
                default:
                    throw new \Exception('Unexpected operation type for shopify bulk operation '.$bulkOperation->operation_type->value);
            }
        }
    }

    public function downloadData(ShopifyBulkOperation $bulkOperation): bool
    {
        $bulkOperation->filename = str_replace('gid://shopify/BulkOperation/', '', $bulkOperation->admin_graphql_api_id).'.jsonl';

        try {
            Storage::disk('shopify')
                ->put(
                    $bulkOperation->filename,
                    file_get_contents($bulkOperation->url)
                );

            $bulkOperation->update();

            return true;
        } catch (Throwable $t) {
            return false;
        }
    }
}
