<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Data\FinancialSummaryQueryData;
use App\DTO\ReportingDailyFinancialDto;
use App\Enums\FinancialMetricEnum;
use App\Helpers;
use App\Models\Product;
use App\Models\ReportingDailyFinancial;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\Setting;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redis;
use Illuminate\Support\LazyCollection;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;

/*
 * TODO: Review unit cost calculations here (Will help with observers)
 * SalesOrderRepository:handleFifoLayerUpdate() (Product Average Cost)
 * SalesOrderRepository:reserveInventoryFromSalesOrderLine() (Product Average Cost)
 * SalesOrderLine:addFBAFifoLayer() (Product Average Cost)
 * ApproveSalesOrderService:approveNormalLines() (Product:getUnitCostAtWarehouse)
 * ApproveSalesOrderService:approveFBALine() (Product Average Cost)
 *
 */

class SalesOrderLineFinancialsRepository extends AbstractRepository
{
    public function saveBulk(Collection $salesOrderLineFinancialsCollection): void
    {
        SalesOrderLineFinancial::dataFeedBulkImport([
            'data_to_import' => [
                'type' => 'json',
                'data' => $salesOrderLineFinancialsCollection->toJson(),
            ],
            'mappings' => [
                [
                    'expected_column_name' => 'sales_order_line_id',
                    'data_column_name' => 'sales_order_line_id',
                    'expected_column_type' => 'integer',
                ],
                [
                    'expected_column_name' => 'revenue',
                    'data_column_name' => 'revenue',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'expected_column_name' => 'revenue_allocated',
                    'data_column_name' => 'revenue_allocated',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'expected_column_name' => 'credits',
                    'data_column_name' => 'credits',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'expected_column_name' => 'cogs',
                    'data_column_name' => 'cogs',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'expected_column_name' => 'cost_allocated',
                    'data_column_name' => 'cost_allocated',
                    'expected_column_type' => 'decimal',
                ],
                [
                    'expected_column_name' => 'cogs_returned',
                    'data_column_name' => 'cogs_returned',
                    'expected_column_type' => 'decimal',
                ],
            ],
            'default_columns' => [
                [
                    'expected_column_name' => 'created_at', // TODO: Jatin how should we handle upserts where we don't want to update created_at on just an update?
                    'default_value' => now(),
                    'expected_column_type' => 'datetime',
                ],
                [
                    'expected_column_name' => 'updated_at',
                    'default_value' => now(),
                    'expected_column_type' => 'datetime',
                ],
            ],
            'unique_by_columns' => [
                'sales_order_line_id',
            ],
        ]);
    }

    public function hasInvalidFinancials(): bool
    {
        return Redis::connection('cache')->sCard(SalesOrderLine::INVALID_FINANCIALS_KEY);
    }

    public function seedMissingSalesOrderLineFinancials(): void
    {
        $query = SalesOrderLine::query()
            ->select('sales_order_lines.id as sales_order_line_id')
            ->leftJoin('sales_order_line_financials', 'sales_order_lines.id', '=', 'sales_order_line_financials.sales_order_line_id')
            ->whereNull('sales_order_line_financials.sales_order_line_id');

        foreach ($query->cursor() as $line) {
            DB::table('sales_order_line_financials')->insert([
                'sales_order_line_id' => $line->sales_order_line_id,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            $this->invalidateForSalesOrderLineIds([$line->sales_order_line_id]);
        }
    }

    public function getInvalidatedLines(?array $sales_order_ids = null, ?int $numLines = 1000): LazyCollection|Collection
    {
        $tempTable = 'temp_'.uniqid();
        $tempTableExists = DB::select("SHOW TABLES LIKE '{$tempTable}'");

        if (empty($tempTableExists)) {
            // Create temporary table
            DB::statement("
            CREATE TEMPORARY TABLE {$tempTable} (
                id bigint(20) unsigned NOT NULL
            ) ENGINE=InnoDB
        ");

            // Fetch all line IDs from Redis and insert into the temporary table
            $lineIds = Redis::connection('cache')->smembers(SalesOrderLine::INVALID_FINANCIALS_KEY);
            $lineIdsChunks = array_chunk($lineIds, 1000);

            foreach ($lineIdsChunks as $chunk) {
                DB::table($tempTable)->insert(array_map(function ($lineId) {
                    return ['id' => $lineId];
                }, $chunk));
            }

            // Create an index on the temporary table for better performance
            if (app()->environment() !== 'testing') {
                DB::statement("CREATE INDEX {$tempTable}_id_index ON {$tempTable} (id)");
            }

        }

        // Retrieve lines from the temporary table with optional sales order ID filtering
        $query = SalesOrderLine::query()
            ->join($tempTable, 'sales_order_lines.id', '=', "{$tempTable}.id")
            ->select('sales_order_lines.*');

        if (! empty($sales_order_ids)) {
            $query->whereIn('sales_order_id', $sales_order_ids);
        }

        $query->limit($numLines);

        return $query->get();
    }

    public function clearInvalidFinancialsCache(Collection $salesOrderLines, ?array $sales_order_ids = null): void
    {
        // If specific sales order ids are passed, just delete sales order lines cache for those
        if (! empty($sales_order_ids)) {
            $query = SalesOrderLine::query()
                ->whereIn('sales_order_id', $sales_order_ids);

            if ($query->count() > 0) {
                Redis::connection('cache')->srem(SalesOrderLine::INVALID_FINANCIALS_KEY, ...$query->pluck('id')->toArray());
            }

            return;
        }

        $query = SalesOrderLine::query()
            ->whereIn('id', $salesOrderLines->pluck('id'));

        if ($query->count() > 0) {
            Redis::connection('cache')->srem(SalesOrderLine::INVALID_FINANCIALS_KEY, ...$query->pluck('id')->toArray());
        } else {
            Redis::connection('cache')->del(SalesOrderLine::INVALID_FINANCIALS_KEY);
        }
    }

    public function getSalesOrderLineFinancialsSummaryForProductDate(Product $product, Carbon $date): ReportingDailyFinancialDto
    {
        $utcDate = $date->copy()->setTimezone('UTC');
        $data = SalesOrderLineFinancial::query()
            ->selectRaw('
                SUM(sales_order_lines.quantity) as quantity,
                COUNT(sales_order_line_financials.id) as num_orders,
                SUM(sales_order_line_financials.revenue) as revenue,
                SUM(sales_order_line_financials.revenue_allocated) as revenue_allocated,
                SUM(sales_order_line_financials.credits) as revenue_credits,
                SUM(sales_order_line_financials.cogs) as cost,
                SUM(sales_order_line_financials.cost_allocated) as cost_allocated,
                SUM(sales_order_line_financials.cogs_returned) as cost_returned
            ')
            ->join('sales_order_lines', 'sales_order_lines.id', '=', 'sales_order_line_financials.sales_order_line_id')
            ->join('sales_orders', 'sales_orders.id', '=', 'sales_order_lines.sales_order_id')
            ->where('sales_orders.order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->where('sales_order_lines.product_id', $product->id)
            ->whereRaw("DATE(CONVERT_TZ(DATE(CONVERT_TZ(sales_orders.order_date, 'UTC', '".Helpers::getAppTimezone()."')), '".Helpers::getAppTimezone()."', 'UTC')) = '{$utcDate->toDateString()}'")
            ->groupBy('sales_order_lines.product_id')
            ->first();

        if (! $data) {
            $reportingDailyFinancialDto = ReportingDailyFinancialDto::from([
                'quantity' => 0,
                'num_orders' => 0,
                'revenue' => 0,
                'revenue_allocated' => 0,
                'revenue_credits' => 0,
                'cost' => 0,
                'cost_allocated' => 0,
                'cost_returned' => 0,
            ]);
        } else {
            $reportingDailyFinancialDto = ReportingDailyFinancialDto::from($data);
        }

        $reportingDailyFinancialDto->date = $date->toDateTimeString();
        $reportingDailyFinancialDto->reportable_id = $product->id;
        $reportingDailyFinancialDto->reportable_type = Product::class;

        return $reportingDailyFinancialDto;
    }

    /**
     * TODO: This is not a full financial report.  It only includes sales order line attributed revenue/costs.
     *
     * @throws Exception
     */
    public function getSummaryByPeriod(string $period, ?int $trailing_days = null): EloquentCollection
    {
        $dateFormat = match ($period) {
            'day' => '%Y-%m-%d',
            'month' => '%Y-%m',
            'year' => '%Y',
            default => throw new Exception('Unknown period in getSummaryByPeriod()'),
        };

        $query = QueryBuilder::for(SalesOrderLine::class)
            ->selectRaw("DATE_FORMAT(CONVERT_TZ(order_date, 'UTC', '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '".$dateFormat."') as order_date")
            ->selectRaw('COUNT(DISTINCT sales_order_lines.sales_order_id) as num_orders')
            ->selectRaw('SUM(CASE WHEN `is_product` = 1 THEN `quantity` ELSE 0 END) as quantity_sold')
            ->selectRaw('SUM(`sales_order_line_financials`.`total_revenue`) as total_revenue')
            ->selectRaw('SUM(
                        (sales_order_line_financials.`cogs` + 
                        sales_order_line_financials.`cost_allocated` - 
                        sales_order_line_financials.`cogs_returned`)
                    ) as total_cost')
            ->join('sales_order_line_financials', 'sales_order_line_financials.sales_order_line_id', 'sales_order_lines.id')
            ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
            ->where('sales_orders.order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->groupBy(DB::raw("DATE_FORMAT(CONVERT_TZ(order_date, 'UTC', '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '".$dateFormat."')"))
            ->allowedFilters([
                // Start Date and End Date are expected to be in User Timezone and need to be converted to UTC in the scope
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
                AllowedFilter::exact('supplier_id', 'supplier_products.supplier_id'),
                AllowedFilter::exact('customer_id', 'sales_orders.customer_id'),
            ]);

        if (Arr::has($_REQUEST, 'filter.supplier_id')) {
            $query
                ->join('products', 'products.id', 'sales_order_lines.product_id')
                ->leftJoin('supplier_products', function (JoinClause $join) {
                    $join->on('supplier_products.product_id', 'sales_order_lines.product_id');
                    $join->where('is_default', 1);
                });
        }

        if (! Arr::has($_REQUEST, 'filter.start_date') && ! Arr::has($_REQUEST, 'filter.end_date') && $trailing_days) {
            $query->trailingDays($trailing_days);
        }

        return $query->get();
    }

    public function getSummaryByProductPeriod(Product $product, string $period, ?int $trailing_days = null): EloquentCollection
    {
        $query = QueryBuilder::for(ReportingDailyFinancial::class)
            ->where('reportable_id', $product->id)
            ->where('reportable_type', Product::class)
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
            ]);

        if (! Arr::has($_REQUEST, 'filter.start_date') && ! Arr::has($_REQUEST, 'filter.end_date') && $trailing_days) {
            $query->trailingDays($trailing_days);
        }

        return $query->get();
    }

    public function getTopProducts(string $metric, int $limit): EloquentCollection
    {
        if (Arr::has($_REQUEST, 'filter.customer_id')) {
            return $this->getTopProductsFromSalesOrderLines($metric, $limit);
        }

        $query = QueryBuilder::for(ReportingDailyFinancial::class)
            ->with(['reportable.primaryImage', 'reportable.brand'])
            ->selectRaw('reporting_daily_financials.reportable_id')
            ->selectRaw('reporting_daily_financials.reportable_type')
            ->selectRaw('SUM(`quantity`) as quantity_sold')
            ->selectRaw('SUM(`total_revenue`) as total_revenue')
            ->selectRaw('SUM(`total_cost`) as total_cost')
            ->selectRaw('SUM(`profit`) as total_profit')
            ->selectRaw('SUM(`num_orders`) as num_orders')
            ->join('products', 'products.id', 'reporting_daily_financials.reportable_id')
            ->where('reporting_daily_financials.reportable_type', Product::class)
            ->leftJoin('product_brands', 'product_brands.id', 'products.brand_id')
            ->groupBy(['products.id', 'reporting_daily_financials.reportable_id', 'reporting_daily_financials.reportable_type'])
            ->orderBy('total_'.$metric, 'DESC')
            ->limit($limit)
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
                AllowedFilter::exact('supplier_id', 'supplier_products.supplier_id'),
            ]);

        if (Arr::has($_REQUEST, 'filter.supplier_id')) {
            $query
                ->leftJoin('supplier_products', function (JoinClause $join) {
                    $join->on('supplier_products.product_id', 'reporting_daily_financials.reportable_id');
                    $join->where('is_default', 1);
                });
        }

        return $query->get();
    }

    public function getTopProductsFromSalesOrderLines(string $metric, int $limit): EloquentCollection
    {
        $query = QueryBuilder::for(SalesOrderLine::class)
            ->selectRaw('sales_order_lines.product_id')
            ->selectRaw('products.sku')
            ->selectRaw('SUM(`quantity`) as quantity_sold')
            ->selectRaw('SUM(`quantity` * (`amount` * sales_orders.currency_rate) - IF(`is_tax_included` = 1, `tax_allocation`, 0) + sales_order_line_financials.revenue_allocated - sales_order_line_financials.credits) as total_revenue')
            ->selectRaw('SUM(
                        (sales_order_line_financials.`cogs` + 
                        sales_order_line_financials.`cost_allocated` - 
                        sales_order_line_financials.`cogs_returned`)
                    ) as total_cost')
            ->selectRaw('SUM(`quantity` * (`amount` * sales_orders.currency_rate) - IF(`is_tax_included` = 1, `tax_allocation`, 0) + sales_order_line_financials.revenue_allocated - sales_order_line_financials.credits)
            - SUM(
                        (sales_order_line_financials.`cogs` + 
                        sales_order_line_financials.`cost_allocated` - 
                        sales_order_line_financials.`cogs_returned`)
                    ) as total_profit')
            ->selectRaw('COUNT(DISTINCT(sales_order_lines.sales_order_id)) as num_orders')
            ->join('sales_order_line_financials', 'sales_order_line_financials.sales_order_line_id', 'sales_order_lines.id')
            ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
            ->join('products', 'products.id', 'sales_order_lines.product_id')
//            ->join('product_brands', 'product_brands.id', 'products.brand_id')
            ->groupBy(['products.sku', 'products.id', 'sales_order_lines.product_id'])
            ->orderBy('total_'.$metric, 'DESC')
            ->limit($limit)
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
                AllowedFilter::exact('customer_id', 'sales_orders.customer_id'),
            ]);

        return $query->get();
    }

    public function seedFromSalesOrderLine(SalesOrderLine $salesOrderLine): void
    {
        $salesOrderLine->salesOrderLineFinancial()->create([
            'sales_order_line_id' => $salesOrderLine->id,
            'revenue' => 0,
            'revenue_allocated' => 0,
            'credits' => 0,
            'cogs' => 0,
            'cogs_returned' => 0,
            'cost_allocated' => 0,
        ]);
    }

    public function invalidateForSalesOrderLineIds(array|int $salesOrderLineIds): void
    {
        $salesOrderLineIds = Arr::wrap($salesOrderLineIds);
        if (count($salesOrderLineIds) > 0) {
            Redis::connection('cache')->sadd(SalesOrderLine::INVALID_FINANCIALS_KEY, ...$salesOrderLineIds);
        }
    }

    public function getFinancialsBySalesChannel(FinancialSummaryQueryData $data): Collection
    {
        $query = QueryBuilder::for(SalesOrderLineFinancial::class)
            ->selectRaw('integration_instances.name as rowLabel')
            ->selectRaw($this->getMetricSelect($data->metric).' as metric')
            ->join('sales_order_lines', 'sales_order_lines.id', 'sales_order_line_financials.sales_order_line_id')
            ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
            ->join('sales_channels', 'sales_channels.id', 'sales_orders.sales_channel_id')
            ->join('integration_instances', 'integration_instances.id', 'sales_channels.integration_instance_id')
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
                AllowedFilter::callback('integration', function (Builder $query, $value) {
                    $query->where('integration_instances.id', $value);
                }),
            ]);

        if ($data->limit) {
            $selfJoinQuery = '('.(clone $query)
                ->groupBy('integration_instances.name')
                ->orderBy('metric', $data->sort->value)
                ->limit($data->limit)
                ->toSqlWithBindings().') as self_joined_table';

            $query->join(DB::raw($selfJoinQuery), function ($join) {
                $join->on('self_joined_table.rowLabel', 'integration_instances.name');
            });
        }

        return $query
            ->selectRaw('DATE_FORMAT(CONVERT_TZ(sales_orders.order_date, "UTC", "' . Helpers::getAppTimezone() . '"), "'.getQueryDateFormat($data->interval->value).'") as `interval`')
            ->orderBy('interval')
            ->groupBy('integration_instances.name', 'interval')
            ->get();
    }

    public function getFinancialsByProductType(FinancialSummaryQueryData $data): Collection
    {
        $query = QueryBuilder::for(SalesOrderLineFinancial::class)
            ->selectRaw("CONCAT(UPPER(SUBSTRING(IF(sales_order_lines.bundle_id IS NOT NULL, 'bundle', products.type), 1, 1)), LOWER(SUBSTRING(IF(sales_order_lines.bundle_id IS NOT NULL, 'bundle', products.type), 2))) as rowLabel")
            ->selectRaw($this->getMetricSelect($data->metric).' as metric')
            ->join('sales_order_lines', 'sales_order_lines.id', 'sales_order_line_financials.sales_order_line_id')
            ->join('products', 'products.id', 'sales_order_lines.product_id')
            ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
            ->join('sales_channels', 'sales_channels.id', 'sales_orders.sales_channel_id')
            ->join('integration_instances', 'integration_instances.id', 'sales_channels.integration_instance_id')
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
                AllowedFilter::callback('integration', function (Builder $query, $value) {
                    $query->where('integration_instances.id', $value);
                }),
            ]);

        if ($data->limit) {
            $selfJoinQuery = '('.(clone $query)
                ->orderBy('metric', $data->sort->value)
                ->groupBy('rowLabel')
                ->limit($data->limit)
                ->toSqlWithBindings().') as self_joined_table';

            $query->join(DB::raw($selfJoinQuery), function ($join) {
                $join->on('self_joined_table.rowLabel', DB::raw("CONCAT(UPPER(SUBSTRING( IF(sales_order_lines.bundle_id IS NOT NULL, 'bundle', products.type), 1, 1)), LOWER(SUBSTRING( IF(sales_order_lines.bundle_id IS NOT NULL, 'bundle', products.type), 2)))"));
            });
        }

        return $query
            ->selectRaw('DATE_FORMAT(CONVERT_TZ(sales_orders.order_date, "UTC", "' . Helpers::getAppTimezone() . '"), "'.getQueryDateFormat($data->interval->value).'") as `interval`')
            ->groupBy('rowLabel', 'interval')
            ->orderBy('interval')
            ->get();
    }

    public function getFinancialsByBrand(FinancialSummaryQueryData $data): Collection
    {
        $query = QueryBuilder::for(SalesOrderLineFinancial::class)
            ->selectRaw("product_brands.name as rowLabel")
            ->selectRaw($this->getMetricSelect($data->metric).' as metric')
            ->join('sales_order_lines', 'sales_order_lines.id', 'sales_order_line_financials.sales_order_line_id')
            ->join('products', 'products.id', 'sales_order_lines.product_id')
            ->join('product_brands', 'product_brands.id', 'products.brand_id')
            ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
            ->join('sales_channels', 'sales_channels.id', 'sales_orders.sales_channel_id')
            ->join('integration_instances', 'integration_instances.id', 'sales_channels.integration_instance_id')
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
                AllowedFilter::callback('integration', function (Builder $query, $value) {
                    $query->where('integration_instances.id', $value);
                }),
            ]);

        if ($data->limit) {
            $selfJoinQuery = '('.(clone $query)
                ->orderBy('metric', $data->sort->value)
                ->groupBy('rowLabel')
                ->limit($data->limit)
                ->toSqlWithBindings().') as self_joined_table';

            $query->join(DB::raw($selfJoinQuery), function ($join) {
                $join->on('self_joined_table.rowLabel', 'product_brands.name');
            });
        }

        return $query
            ->selectRaw('DATE_FORMAT(CONVERT_TZ(sales_orders.order_date, "UTC", "' . Helpers::getAppTimezone() . '"), "'.getQueryDateFormat($data->interval->value).'") as `interval`')
            ->groupBy('rowLabel', 'interval')
            ->orderBy('interval')
            ->get();
    }

    public function getFinancialsBySku(FinancialSummaryQueryData $data): Collection
    {
        $query = QueryBuilder::for(SalesOrderLineFinancial::class)
            ->selectRaw("products.sku as rowLabel, suppliers.name as Supplier, product_brands.name as Brand, products.name as Name")
            ->selectRaw($this->getMetricSelect($data->metric).' as metric')
            ->join('sales_order_lines', 'sales_order_lines.id', 'sales_order_line_financials.sales_order_line_id')
            ->join('products', 'products.id', 'sales_order_lines.product_id')
            ->join('product_brands', 'product_brands.id', 'products.brand_id')
            ->leftJoin('supplier_products', 'supplier_products.product_id', '=', 'sales_order_lines.product_id')
            ->leftJoin('suppliers', 'suppliers.id', '=', 'supplier_products.supplier_id')
            ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
            ->join('sales_channels', 'sales_channels.id', 'sales_orders.sales_channel_id')
            ->join('integration_instances', 'integration_instances.id', 'sales_channels.integration_instance_id')
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
                AllowedFilter::callback('integration', function (Builder $query, $value) {
                    $query->where('integration_instances.id', $value);
                }),
            ]);

        if ($data->limit) {
            $selfJoinQuery = '('.(clone $query)
                    ->orderBy('metric', $data->sort->value)
                    ->groupBy('rowLabel')
                    ->limit($data->limit)
                    ->toSqlWithBindings().') as self_joined_table';

            $query->join(DB::raw($selfJoinQuery), function ($join) {
                $join->on('self_joined_table.rowLabel', 'products.sku');
            });
        }

        return $query
            ->selectRaw('DATE_FORMAT(CONVERT_TZ(sales_orders.order_date, "UTC", "' . Helpers::getAppTimezone() . '"), "'.getQueryDateFormat($data->interval->value).'") as `interval`')
            ->groupBy('rowLabel', 'Supplier', 'Brand', 'Name', 'interval')
            ->orderBy('interval')
            ->get();
    }

    public function getFinancialsBySupplier(FinancialSummaryQueryData $data): Collection
    {
        $query = QueryBuilder::for(SalesOrderLineFinancial::class)
            ->selectRaw("suppliers.name as rowLabel")
            ->selectRaw($this->getMetricSelect($data->metric).' as metric')
            ->join('sales_order_lines', 'sales_order_lines.id', 'sales_order_line_financials.sales_order_line_id')
            ->join('supplier_products', 'supplier_products.product_id', '=', 'sales_order_lines.product_id')
            ->join('suppliers', 'suppliers.id', '=', 'supplier_products.supplier_id')
            ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
            ->join('sales_channels', 'sales_channels.id', 'sales_orders.sales_channel_id')
            ->join('integration_instances', 'integration_instances.id', 'sales_channels.integration_instance_id')
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
                AllowedFilter::callback('integration', function (Builder $query, $value) {
                    $query->where('integration_instances.id', $value);
                }),
            ]);

        if ($data->limit) {
            $selfJoinQuery = '('.(clone $query)
                ->orderBy('metric', $data->sort->value)
                ->groupBy('rowLabel')
                ->limit($data->limit)
                ->toSqlWithBindings().') as self_joined_table';

            $query->join(DB::raw($selfJoinQuery), function ($join) {
                $join->on('self_joined_table.rowLabel', 'suppliers.name');
            });
        }

        return $query
            ->selectRaw('DATE_FORMAT(CONVERT_TZ(sales_orders.order_date, "UTC", "' . Helpers::getAppTimezone() . '"), "'.getQueryDateFormat($data->interval->value).'") as `interval`')
            ->groupBy('rowLabel', 'interval')
            ->orderBy('interval')
            ->get();
    }

    private function getMetricSelect(FinancialMetricEnum $metric): string
    {
        return match ($metric) {
            FinancialMetricEnum::REVENUE => 'SUM(sales_order_line_financials.total_revenue)',
            FinancialMetricEnum::PROFIT => 'SUM(sales_order_line_financials.profit)',
            FinancialMetricEnum::UNITS_SOLD => 'SUM(IF(sales_order_lines.is_product = 1, sales_order_lines.quantity, 0))',
            FinancialMetricEnum::NUM_ORDERS => 'COUNT(DISTINCT sales_order_lines.sales_order_id)',
        };
    }
}
