<?php

namespace App\Repositories;

use App\Collections\SalesCreditLineCollection;
use App\DTO\SalesCreditDto;
use App\DTO\SalesCreditLineDto;
use App\Models\ReturnReason;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturn;

class SalesCreditRepository
{
    public function receiveAll(SalesCredit $salesCredit, string $action, string $reason, string $notes = ''): void
    {
        $salesCreditReturn = new SalesCreditReturn([
            'warehouse_id' => $salesCredit->to_warehouse_id,
            'shipped_at' => $salesCredit->credit_date,
            'received_at' => $salesCredit->credit_date,
        ]);
        $salesCredit->salesCreditReturns()->save($salesCreditReturn);

        /** @var ReturnReason $returnReason */
        $returnReason = ReturnReason::query()->firstOrCreate([
            'name' => $reason,
        ]);

        // add its lines and add to inventory
        $salesCreditReturn->setReturnLines(
            $salesCredit->salesCreditLines()
                ->get()
                ->map(function (SalesCreditLine $salesCreditLine) use ($action, $returnReason, $notes) {
                    return [
                        'sales_credit_line_id' => $salesCreditLine->id,
                        'quantity' => $salesCreditLine->quantity,
                        'product_id' => $salesCreditLine->product_id ?? null,
                        'action' => $action,
                        'reason_id' => $returnReason->id,
                        'notes' => $notes,
                    ];
                })->toArray()
        );

        $salesCreditReturn->save();

        // mark sales credit as returned
        $salesCredit->returned($salesCreditReturn->received_at);
    }

    public function create(SalesCreditDto $salesCreditDto): SalesCredit
    {
        $salesCredit = new SalesCredit($salesCreditDto->toArray());
        $salesCredit->save();

        $this->saveLines($salesCredit, $salesCreditDto->sales_credit_lines);

        return $salesCredit;
    }

    private function saveLines(SalesCredit $salesCredit, SalesCreditLineCollection $salesCreditLines): void
    {
        $salesCreditLines->each(function (SalesCreditLineDto $salesCreditLineDto) use ($salesCredit) {
            $salesCreditLine = new SalesCreditLine($salesCreditLineDto->toArray());
            $salesCreditLine->salesCredit()->associate($salesCredit);
            $salesCreditLine->save();
        });
    }
}
