<?php

namespace App\Repositories;

use App\Data\FinancialSummaryQueryData;
use App\Enums\FinancialMetricEnum;
use App\Helpers;
use App\Models\Product;
use App\Models\ReportingDailyFinancial;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;
use Str;

class ReportingDailyFinancialRepository
{
    protected ?string $userTimezone;

    public function __construct()
    {
        $this->userTimezone = Helpers::getAppTimezone();
    }

    public function getFinancialsByBrand(FinancialSummaryQueryData $data): Collection
    {
        return $this->getFinancials('product_brands', 'brand_id', 'product_brands.name', $data);
    }

    public function getFinancialsBySupplier(FinancialSummaryQueryData $data): Collection
    {
        return $this->getFinancials('supplier_products', 'supplier_id', 'suppliers.name', $data, [
            [
                'column' => 'supplier_products.is_default',
                'value' => 1,
            ]
        ]);
    }

    public function getFinancialsBySku(FinancialSummaryQueryData $data): Collection
    {
        return $this->getFinancials('products', 'id', 'products.sku', $data, [], [
            'Name' => 'products.name',
            'Brand' => 'product_brands.name',
            'Supplier' => 'suppliers.name',
        ]);
    }

    private function getFinancials(
        string $tableName,
        string $tableId,
        string $selectRaw,
        FinancialSummaryQueryData $data,
        array $additionalFilters = [],
        array $additionalSelects = []
    ): Collection
    {
        $query = $this->getFinancialsQuery($data)
            ->selectRaw($selectRaw.' as rowLabel')
            ->orderBy('metric', $data->sort ? $data->sort->value : 'asc');

        if ($additionalSelects) {
            foreach ($additionalSelects as $key => $value) {
                $query->addSelect(DB::raw($value.' as '.$key));
            }
        }

        $tableJoiners = [
            'products' => function ($query) {
                $query->join('product_brands', 'product_brands.id', '=', 'products.brand_id');
                $query->leftJoin('supplier_products', function (JoinClause $join) {
                    $join->on('supplier_products.product_id', '=', 'products.id');
                    $join->where('supplier_products.is_default', '=', 1);
                });
                $query->leftJoin('suppliers', 'suppliers.id', '=', 'supplier_products.supplier_id');
            },
            'supplier_products' => function ($query) {
                $query->join('supplier_products', 'supplier_products.product_id', '=', 'products.id');
                $query->join('suppliers', 'suppliers.id', '=', 'supplier_products.supplier_id');
            },
            'default' => function ($query, $tableName, $tableId) {
                $query->join($tableName, $tableName.'.id', '=', 'products.'.$tableId);
            },
        ];

        $tableJoiner = $tableJoiners[$tableName] ?? $tableJoiners['default'];
        $tableJoiner($query, $tableName, $tableId);

        if (!empty($additionalFilters))
        {
            foreach ($additionalFilters as $filter) {
                $query->where($filter['column'], $filter['value']);
            }
        }

        // Get top records based on aggregate function
        if ($data->limit) {
            $joinSubQuery = '('.Str::replace('\\', '\\\\', (clone $query)->groupBy($selectRaw)->toSqlWithBindings())." LIMIT $data->limit) as self_joined_table";

            $query->join(DB::raw($joinSubQuery), function ($join) use ($selectRaw) {
                $join->on('self_joined_table.rowLabel', $selectRaw);
            });
        }

        return $query
            ->groupBy($selectRaw, DB::raw("DATE_FORMAT(CONVERT_TZ(reporting_daily_financials.date, 'UTC', '" . $this->userTimezone . "'), '".getQueryDateFormat($data->interval->value)."')"))
            ->orderBy('interval')
            ->get();
    }

    public function getFinancialsQuery(FinancialSummaryQueryData $data): QueryBuilder
    {
        return QueryBuilder::for(ReportingDailyFinancial::class)
            ->selectRaw("DATE_FORMAT(CONVERT_TZ(reporting_daily_financials.date, 'UTC', '" . $this->userTimezone . "'), '".getQueryDateFormat($data->interval->value)."') as `interval`")
            ->selectRaw($this->getMetricSelect($data->metric).' as metric')
            ->join('products', 'products.id', '=', 'reporting_daily_financials.reportable_id')
            ->where('reporting_daily_financials.reportable_type', Product::class)
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
            ]);
    }

    private function getMetricSelect(FinancialMetricEnum $metric): string
    {
        return match ($metric) {
            FinancialMetricEnum::REVENUE => 'SUM(reporting_daily_financials.total_revenue)',
            FinancialMetricEnum::PROFIT => 'SUM(reporting_daily_financials.profit)',
            FinancialMetricEnum::UNITS_SOLD => 'SUM(reporting_daily_financials.quantity)',
            FinancialMetricEnum::NUM_ORDERS => 'SUM(reporting_daily_financials.num_orders)',
        };
    }
}
