<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Data\CreateDropshipPurchaseOrderData;
use App\DTO\PurchaseOrderDto;
use App\DTO\PurchaseOrderLineDto;
use App\Exceptions\NegativeInventoryFulfilledSalesOrderLinesException;
use App\Models\OrderLink;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipment;
use App\Models\PurchaseOrderShipmentLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Services\InventoryManagement\BulkInventoryManager;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Throwable;

class PurchaseOrderRepository extends AbstractRepository
{
    /**
     * @throws Throwable
     */
    public function saveWithRelations(#[DataCollectionOf(PurchaseOrderDto::class)] DataCollection $data): Collection|PurchaseOrder
    {
        $data = $data->toCollection();

        return DB::transaction(function () use ($data) {
            $purchaseOrderCollection = $data->map(function (PurchaseOrderDto $purchaseOrderDto) {
                return PurchaseOrderDto::from(new PurchaseOrder($purchaseOrderDto->toArray()));
            });


            $purchaseOrderCollection = $this->save($purchaseOrderCollection, PurchaseOrder::class);
            $this->saveLines($purchaseOrderCollection, $data);

            if ($purchaseOrderCollection->count() === 1) {
                return PurchaseOrder::find($purchaseOrderCollection->first()['id']);
            }

            return PurchaseOrder::with([
                'purchaseOrderLines.product'
            ])->whereIn('id', $purchaseOrderCollection->pluck('id'))->get();
        });
    }

    private function saveLines(Collection $purchaseOrderCollection, $data): void
    {
        if ($purchaseOrderCollection->isEmpty()) {
            return;
        }

        $purchaseOrderLineCollection = collect();

        $purchaseOrderCollection->each(function ($purchaseOrder) use ($data, $purchaseOrderLineCollection) {
            $purchaseOrderDto = $data->where('purchase_order_number', $purchaseOrder['purchase_order_number'])->first();
            $purchaseOrderDto->purchaseOrderLines->each(function (PurchaseOrderLineDto $purchaseOrderLineDto) use (
                $purchaseOrderDto,
                $purchaseOrder,
                $purchaseOrderLineCollection
            ) {
                $purchaseOrderLineDto->purchase_order_id = $purchaseOrder['id'];
                $purchaseOrderLineCollection->add(PurchaseOrderLineDto::from(new PurchaseOrderLine($purchaseOrderLineDto->toArray())));
            });
        });

        $this->save($purchaseOrderLineCollection, PurchaseOrderLine::class);
    }

    public function createDropshipPurchaseOrder(
        SalesOrder $salesOrder,
        Warehouse $warehouse,
        Collection $salesOrderLines
    ): PurchaseOrder {
        $purchaseOrder = new PurchaseOrder([
            'purchase_order_date' => $salesOrder->order_date,
            'order_status' => PurchaseOrder::STATUS_OPEN,
            'sales_order_id' => $salesOrder->id,
            'requested_shipping_method_id' => $salesOrder->shipping_method_id,
            'submission_format' => $warehouse->supplier->purchase_order_format ??
                SettingRepository::getSettingByKey(Setting::KEY_PO_FORMAT),
            'supplier_id' => $warehouse->supplier_id,
            'supplier_warehouse_id' => $warehouse->id,
            'destination_address_id' => $salesOrder->shipping_address_id,
            'currency_id' => $salesOrder->currency_id,
        ]);

        // Dropship PO has receipt status of dropship as it
        // will be sent directly to the customer on the sales order.
        $purchaseOrder->receipt_status = PurchaseOrder::RECEIPT_STATUS_DROPSHIP;

        $purchaseOrderLines = $salesOrderLines->map(function (SalesOrderLine $line) {
            return $line->only([
                'description',
                'product_id',
                'amount',
                'quantity',
            ]);
        });

        $salesOrder->purchaseOrders()->save($purchaseOrder);

        $purchaseOrder->setPurchaseOrderLines($purchaseOrderLines->toArray());
        $purchaseOrder->load('purchaseOrderLines');

        // Create record in order links
        // TODO: Order links relation may be removed in the future.
        $orderLink = new OrderLink();
        $orderLink->child = $purchaseOrder;
        $orderLink->link_type = OrderLink::LINK_TYPE_DROPSHIP;
        $salesOrder->childLinks()->save($orderLink);

        return $purchaseOrder;
    }

    public function findDropshipOrder(SalesOrder $salesOrder, Warehouse $warehouse): PurchaseOrder|Model|null
    {
        return $salesOrder->purchaseOrders()
            ->where('supplier_warehouse_id', $warehouse->id)
            ->with(['purchaseOrderLines'])
            ->first();
    }

    public function createShipment(PurchaseOrder $purchaseOrder, array $payload): PurchaseOrderShipment
    {
        /** @var PurchaseOrderShipment $shipment */
        $shipment = $purchaseOrder->purchaseOrderShipments()->create($payload);

        // Add the shipment lines
        $shipment->purchaseOrderShipmentLines()->createMany(
            $payload['purchase_order_shipment_lines']
        );

        return $shipment->load('purchaseOrderShipmentLines')->refresh();
    }

    public function suppliersEligibleForBackorderPurchasing(): EloquentCollection
    {
        return Supplier::with([
            'backorderQueues',
            'backorderQueues.salesOrderLine',
            'backorderQueues.backorderQueueReleases',
            'backorderQueues.backorderQueueCoverages',
        ])
            ->whereHas('backorderQueues', function (Builder $builder) {
                $builder->whereColumn('backordered_quantity', '>', 'released_quantity');
                // We only want to consider suppliers with uncovered backorders because if covered, it means there is already incoming stock
                $builder->whereHas('backorderQueueCoverages', function (Builder $builder) {
                    $builder->select(DB::raw('IFNULL(SUM(`covered_quantity`), 0)'));
                }, '<', DB::raw('backordered_quantity'));
            })
            ->where('auto_generate_backorder_po', true)
            ->get();
    }

    public function createPurchaseOrder(PurchaseOrderDto|CreateDropshipPurchaseOrderData $purchaseOrderDto): PurchaseOrder
    {
        $purchaseOrder = new PurchaseOrder($purchaseOrderDto->toArray());
        $purchaseOrder->save();

        $purchaseOrder->setPurchaseOrderLines($this->aggregateLinesByProduct($purchaseOrderDto->purchaseOrderLines->toArray(), $purchaseOrder));

        return $purchaseOrder;
    }

    protected function aggregateLinesByProduct(array $lines, PurchaseOrder $purchaseOrder): array
    {
        $lines = collect($lines);
        $byProduct = $lines->groupBy('product_id')->toArray();
        $aggregated = [];
        foreach ($byProduct as $productId => $productLines) {
            $line = $productLines[0];
            $product = Product::find($line['product_id']);
            $amount = $line['amount'] ?? app(ProductRepository::class)
                ->getCostForSupplier($product, $purchaseOrder->destination_warehouse_id, $purchaseOrder->supplier_id);
            $aggregated[] = array_merge($line, [
                'quantity' => collect($productLines)->sum('quantity'),
                'amount' => $amount,
            ]);
        }

        return $aggregated;
    }

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     */
    public function bulkDelete(Collection $purchaseOrders): void
    {
        $purchaseOrderIds = $purchaseOrders->pluck('id');

        $purchaseOrderShipmentReceiptLines = PurchaseOrderShipmentReceiptLine::query()
            ->whereHas('purchaseOrderShipmentReceipt.purchaseOrderShipment', function (Builder $builder) use ($purchaseOrderIds) {
                $builder->whereIn('purchase_order_id', $purchaseOrderIds);
            })
            ->get();

        (new BulkInventoryManager())->bulkDeletePositiveInventoryEvents($purchaseOrderShipmentReceiptLines);

        PurchaseOrderShipmentReceipt::query()
            ->whereHas('purchaseOrderShipment', function (Builder $builder) use ($purchaseOrderIds) {
                $builder->whereIn('purchase_order_id', $purchaseOrderIds);
            })
            ->delete();

        PurchaseOrderShipmentLine::query()
            ->whereHas('purchaseOrderLine', function (Builder $builder) use ($purchaseOrderIds) {
                $builder->whereIn('purchase_order_id', $purchaseOrderIds);
            })
            ->delete();

        PurchaseOrderShipment::query()
            ->whereIn('purchase_order_id', $purchaseOrderIds)
            ->delete();

        PurchaseOrderLine::query()
            ->whereIn('purchase_order_id', $purchaseOrderIds)
            ->delete();

        PurchaseOrder::query()
            ->whereIn('id', $purchaseOrderIds)
            ->delete();
    }

    public function getUnreceivedPurchaseOrderLinesForOpenPurchaseOrders(?array $purchaseOrderLineIds = null, ?array $productIds = null, ?int $warehouseId = null): EloquentCollection
    {
        $query = PurchaseOrderLine::query()
            ->selectRaw('purchase_order_lines.id, purchase_order_lines.quantity, purchase_order_lines.quantity - SUM(IFNULL(purchase_order_shipment_receipt_lines.quantity, 0)) unreceived_quantity, purchase_orders.estimated_delivery_date, purchase_orders.purchase_order_date, product_id, purchase_orders.destination_warehouse_id')
            ->join('purchase_orders', 'purchase_orders.id', '=', 'purchase_order_lines.purchase_order_id')
            ->where('purchase_orders.order_status', '=', 'open')
            ->whereNotNull('purchase_order_lines.product_id')
            ->whereNotNull('purchase_orders.destination_warehouse_id')
            ->leftJoin('purchase_order_shipment_lines', 'purchase_order_shipment_lines.purchase_order_line_id', '=', 'purchase_order_lines.id')
            ->leftJoin('purchase_order_shipment_receipt_lines', 'purchase_order_shipment_receipt_lines.purchase_order_shipment_line_id', '=', 'purchase_order_shipment_lines.id')
            ->groupBy('purchase_order_lines.id')
            ->havingRaw('unreceived_quantity > 0');

        if (! empty($purchaseOrderLineIds)) {
            $query->whereIn('purchase_order_lines.id', $purchaseOrderLineIds);
        }

        if (! empty($productIds)) {
            $query->whereIn('purchase_order_lines.product_id', $productIds);
        }

        if (! empty($warehouseId)) {
            $query->where('purchase_orders.destination_warehouse_id', $warehouseId);
        }

        return $query->get();
    }

    public function getUnreceivedPurchaseOrderLinesForOpenPurchaseOrdersProductWarehousePairs(?array $purchaseOrderLineIds = null, ?array $productIds = null, ?int $warehouseId = null): EloquentCollection
    {
        $subquery = PurchaseOrderShipmentLine::query()
            ->select('purchase_order_shipment_lines.purchase_order_line_id')
            ->selectRaw('SUM(IFNULL(purchase_order_shipment_receipt_lines.quantity, 0)) as receipt_quantity')
            ->leftJoin('purchase_order_shipment_receipt_lines', 'purchase_order_shipment_receipt_lines.purchase_order_shipment_line_id', '=', 'purchase_order_shipment_lines.id')
            ->groupBy('purchase_order_shipment_lines.purchase_order_line_id');

        $query = PurchaseOrderLine::query()
            ->select('purchase_order_lines.product_id', 'purchase_orders.destination_warehouse_id')
            ->selectRaw('SUM(purchase_order_lines.quantity) as quantity')
            ->join('purchase_orders', 'purchase_orders.id', '=', 'purchase_order_lines.purchase_order_id')
            ->leftJoinSub($subquery, 'receipt_lines', 'receipt_lines.purchase_order_line_id', '=', 'purchase_order_lines.id')
            ->where('purchase_orders.order_status', '=', 'open')
            ->whereNotNull('purchase_order_lines.product_id')
            ->whereNotNull('purchase_orders.destination_warehouse_id')
            ->whereRaw('IFNULL(receipt_lines.receipt_quantity, 0) < purchase_order_lines.quantity')
            ->groupBy('purchase_order_lines.product_id', 'purchase_orders.destination_warehouse_id');

        //        $query = PurchaseOrderLine::query()
        //            ->select('product_id', 'purchase_orders.destination_warehouse_id', 'purchase_order_lines.quantity')
        //            ->join('purchase_orders', 'purchase_orders.id', '=', 'purchase_order_lines.purchase_order_id')
        //            ->where('purchase_orders.order_status', '=', 'open')
        //            ->whereNotNull('purchase_order_lines.product_id')
        //            ->whereNotNull('purchase_orders.destination_warehouse_id')
        //            ->leftJoin('purchase_order_shipment_lines', 'purchase_order_shipment_lines.purchase_order_line_id', '=', 'purchase_order_lines.id')
        //            ->leftJoin('purchase_order_shipment_receipt_lines', 'purchase_order_shipment_receipt_lines.purchase_order_shipment_line_id', '=', 'purchase_order_shipment_lines.id')
        //            ->groupBy(['product_id', 'destination_warehouse_id', 'purchase_order_lines.quantity'])
        //            ->havingRaw('SUM(IFNULL(purchase_order_shipment_receipt_lines.quantity, 0)) < purchase_order_lines.quantity');

        if (! empty($purchaseOrderLineIds)) {
            $query->whereIn('purchase_order_lines.id', $purchaseOrderLineIds);
        }

        if (! empty($productIds)) {
            $query->whereIn('purchase_order_lines.product_id', $productIds);
        }

        if (! is_null($warehouseId)) {
            $query->where('purchase_orders.destination_warehouse_id', $warehouseId);
        }

        return $query->get();
    }

    public function getProductQuantityInboundForSupplierAndWarehouse(Supplier $supplier, Warehouse $warehouse): EloquentCollection
    {
        return PurchaseOrderLine::query()
            ->select('product_id')
            ->selectRaw('SUM(purchase_order_lines.quantity) AS quantity')
            //->selectRaw('SUM(IFNULL(purchase_order_shipment_receipt_lines.quantity, 0)) AS total_shipment_quantity')
            ->join('purchase_orders', 'purchase_orders.id', '=', 'purchase_order_lines.purchase_order_id')
            ->where('purchase_orders.order_status', '=', 'open')
            ->where('supplier_id', $supplier->id)
            ->where('purchase_orders.destination_warehouse_id', $warehouse->id)
            ->whereNotNull('purchase_order_lines.product_id')
            ->whereNotNull('purchase_orders.destination_warehouse_id')
            ->leftJoin('purchase_order_shipment_lines', 'purchase_order_shipment_lines.purchase_order_line_id', '=', 'purchase_order_lines.id')
            ->leftJoin('purchase_order_shipment_receipt_lines', 'purchase_order_shipment_receipt_lines.purchase_order_shipment_line_id', '=', 'purchase_order_shipment_lines.id')
            ->groupBy('purchase_order_lines.product_id')
            ->havingRaw('SUM(IFNULL(purchase_order_shipment_receipt_lines.quantity, 0)) < SUM(purchase_order_lines.quantity)')
            ->get();
    }
}
